import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {Column, OrderBy, OrderConst} from './constants';
import classNames from 'classnames';
import {Checkbox} from '../checkbox/checkbox.component';

export class InfiniteListHeaderComponent extends Component {

    static propTypes = {
        columns: PropTypes.arrayOf(Column).isRequired,
        selectedRows: PropTypes.arrayOf(PropTypes.string).isRequired,
        rows: PropTypes.arrayOf(PropTypes.any).isRequired,
        onChangeOrder: PropTypes.func.isRequired,
        orderBy: OrderBy,
        toggleSelectAll: PropTypes.func.isRequired,
    };

    getNextDirection(propertyName) {
        const {orderBy} = this.props;
        if (orderBy.propertyName !== propertyName) {
            return OrderConst.SORT_DESC;
        } else {
            return orderBy.direction === OrderConst.SORT_DESC ? OrderConst.SORT_ASC : OrderConst.SORT_DESC;
        }
    }

    onChangeOrder(column) {
        this.props.onChangeOrder(column.propertyName, this.getNextDirection(column.propertyName));
    };

    renderOrderIcon() {
        const {orderBy} = this.props;
        const orderedColumnClassNames = classNames('xl-icon icon-s', {
            'arrow-up-icon': orderBy.direction === OrderConst.SORT_ASC,
            'arrow-down-icon': orderBy.direction === OrderConst.SORT_DESC,
        });

        return (
            <div className="order-icons">
                <i className={orderedColumnClassNames}/>
            </div>
        );
    }

    renderDefaultIcon() {
        return (
            <div className="order-icons order-default">
                <i className="xl-icon icon-s arrow-up-icon"/>
                <i className="xl-icon icon-s arrow-down-icon"/>
            </div>
        )
    }

    renderHeaderCell(column) {
        const {orderBy} = this.props;

        return (
            <div
                key={column.propertyName}
                className={classNames(`list-row-cell ${column.class}`, {'sortable': column.sortable})}
                onClick={() => column.sortable && this.onChangeOrder(column)}>
                {column.label}
                {column.sortable && orderBy && (column.propertyName === orderBy.propertyName
                        ? this.renderOrderIcon()
                        : this.renderDefaultIcon()
                )}
            </div>
        );
    }

    render() {
        const {
            columns,
            toggleSelectAll,
            rows,
            selectedRows,
        } = this.props;
        return (
            <div className="list-header">
                <Checkbox
                    onChange={toggleSelectAll}
                    checked={rows.length === selectedRows.length}
                    isIndeterminate={selectedRows.length > 0 && selectedRows.length < rows.length}
                />
                {columns.map(column => this.renderHeaderCell(column))}
            </div>
        );
    }
}
