import React, {Component} from 'react';
import PropTypes from 'prop-types';
import debounce from 'lodash/debounce';

export class InfiniteListBodyComponent extends Component {

    static propTypes = {
        rows: PropTypes.arrayOf(PropTypes.any).isRequired,
        selectedRows: PropTypes.arrayOf(PropTypes.string).isRequired,
        renderRow: PropTypes.func.isRequired,
        loadMore: PropTypes.func.isRequired,
        toggleSelectedRows: PropTypes.func.isRequired,
        onScrollList: PropTypes.func,
    };

    constructor(props) {
        super(props);
        this.rowsRef = React.createRef();
    }

    componentDidMount() {
        this.rowsRef.current.addEventListener('scroll', this.loadMore);
        this.loadMore();
    }

    componentWillUnmount() {
        this.rowsRef.current.removeEventListener('scroll', this.loadMore);
    }

    loadMore = () => {
        const rowsRef = this.rowsRef.current;
        if (rowsRef.scrollTop + rowsRef.clientHeight >= rowsRef.scrollHeight) {
            this.debouncedLoadMore();
        }
    };

    // TODO this should be debounced on saga, need to update first REL-9148
    debouncedLoadMore = debounce(this.props.loadMore, 500);

    renderRow(row) {
        return (
            <span className="list-row-view-container" key={row.id}>
                <input
                    checked={this.props.selectedRows.includes(row.id)}
                    type="checkbox"
                    onChange={() => this.props.toggleSelectedRows([row])}
                />
                {this.props.renderRow(row)}
            </span>
        )
    }

    render() {
        const {rows, onScrollList} = this.props;
        return (
            <div className="list-view-rows" ref={this.rowsRef} onScroll={onScrollList}>
                {rows.map((row) => this.renderRow(row))}
            </div>
        );
    }
}
