import React from 'react';
import '@testing-library/jest-dom';
import moment from 'moment';
import { act, renderWithStoreAndTheme, screen } from '@xlr-ui/tests/unit/testing-utils';
import { DrawerHistoryRow, DrawerHistoryRowProps } from './drawer-history-row.component';
import { mockReleaseActivityLogs } from '../../../__mocks__/index.mock';
import * as helpers from '@xlr-ui/app/features/common/helper';
import { getActivityTypeLabel } from '../../../helper';

const formatDateSpy = jest.spyOn(helpers, 'formatDate');

describe('DrawerHistoryRow', () => {
    const defaultLog = mockReleaseActivityLogs[0];
    const defaultProps: DrawerHistoryRowProps = {
        log: defaultLog,
    };
    const { username, fullName } = defaultProps.log.user;
    const dateFormat = 'MM/DD/YYYY';

    const getTooltipByTitle = (title: string) => screen.getByRole('tooltip', { name: title });
    const getAvatarByAriaLabel = (ariaLabel: string) => screen.getByRole('img', { name: `Avatar for ${ariaLabel}` });
    const getTypographyByText = (text: string) => screen.getByText(text);

    const renderComponent = (props = defaultProps) => {
        renderWithStoreAndTheme(<DrawerHistoryRow {...props} />);
    };

    beforeEach(() => {
        formatDateSpy.mockImplementation((date) => moment(date).format(dateFormat));
    });

    afterAll(() => {
        jest.clearAllMocks();
    });

    it('should render correct elements', () => {
        renderComponent();
        expect(getTooltipByTitle(username)).toBeVisible();
        const avatar = getAvatarByAriaLabel(username);
        expect(avatar).toBeVisible();
        expect(avatar).toHaveClass('dot-avatar dot-size-small');
        expect(screen.getByText('RA')).toBeVisible();
        const typography = getTypographyByText(fullName as string);
        expect(typography).toBeVisible();
        expect(typography).toHaveClass('MuiTypography-body1');
        expect(getTooltipByTitle(moment(defaultLog.eventTime).format(dateFormat))).toBeVisible();
        const dateAgoTypography = getTypographyByText(moment(defaultLog.eventTime).fromNow());
        expect(dateAgoTypography).toBeVisible();
        expect(dateAgoTypography).toHaveClass('content-right-header-time');
        const pill = screen.getByText(getActivityTypeLabel(defaultLog.activityType));
        expect(pill).toBeVisible();
        expect(pill).toHaveClass('MuiChip-label');
        expect(getTypographyByText(defaultLog.message)).toBeVisible();
    });

    it('should render pill with proper colors', () => {
        const message = 'your task task failed';
        const activityType = 'TASK_FAILED';
        renderComponent({ ...defaultProps, log: { ...defaultLog, activityType, message } });
        const pill = screen.getByText(getActivityTypeLabel(activityType));
        expect(pill).toBeVisible();
        expect(pill.parentElement).toHaveAttribute('bordercolor', '#d61f21');
        expect(pill.parentElement).toHaveAttribute('backgroundcolor', '#fae4e4');
    });

    it('should render with markdown', async () => {
        await act(async () => {
            renderComponent({ ...defaultProps, log: { ...defaultLog, activityType: 'TRIGGER_EXECUTED', message: `Link: <a href="#/releases/">My link</a>` } });
        });
        expect(getTypographyByText('Link: My link')).toBeVisible();
    });
});
