import React, { KeyboardEvent, useState } from 'react';
import { DotIconButton, DotInputText } from '@digital-ai/dot-components';
import classNames from 'classnames';
import './input-with-send.component.less';

export interface InputWithSendProps {
    ai?: boolean;
    autoFocus?: boolean;
    className?: string;
    disabled?: boolean;
    helperText?: string;
    id: string;
    isSending?: boolean;
    maxRows?: number;
    multiline?: boolean;
    name: string;
    onSendClick: (value: string) => void;
    onStopClick?: () => void;
    placeholder?: string;
}

export const InputWithSend = ({
    ai,
    autoFocus,
    className,
    disabled,
    helperText,
    id,
    isSending = false,
    maxRows = 8,
    multiline = true,
    name,
    onSendClick,
    onStopClick,
    placeholder,
}: InputWithSendProps) => {
    const [internalValue, setInternalValue] = useState('');
    const internalValueTrimmed = internalValue.trim();

    const handleKeyDown = (event: KeyboardEvent) => {
        if (event.key === 'Enter' && !event.shiftKey) {
            event.preventDefault();
            handleSend();
        }
    };

    const handleSend = () => {
        if (!internalValueTrimmed) return;
        onSendClick(internalValue); // send the whole value not trimmed, this might be changed in future if needed
        setInternalValue('');
    };

    const handleStop = () => {
        onStopClick?.();
    };

    const endIcon = isSending ? (
        <DotIconButton
            className="input-with-send-cancel-button"
            color="ai"
            data-testid="input-with-send-cancel-button"
            disabled={!isSending}
            iconId="abort"
            onClick={handleStop}
            shape="square"
            size="small"
        />
    ) : (
        <DotIconButton
            color="ai"
            data-testid="input-with-send-button"
            disabled={!internalValueTrimmed}
            iconId="send-airplane"
            onClick={handleSend}
            shape="square"
            size="small"
        />
    );

    const isInputTextDisabled = isSending || disabled;

    return (
        <DotInputText
            ai={ai}
            autoFocus={autoFocus}
            className={classNames('input-with-send', className)}
            disabled={isInputTextDisabled}
            endIcon={endIcon}
            helperText={helperText}
            id={id}
            key={String(isInputTextDisabled)} // Force re-mount to keep autofocus when disabled state changes
            maxRows={maxRows}
            multiline={multiline}
            name={name}
            onChange={(e) => setInternalValue(e.target.value)}
            onKeyDown={handleKeyDown}
            placeholder={placeholder}
            value={internalValue}
        />
    );
};
