import React from 'react';
import '@testing-library/jest-dom';
import { renderWithStoreAndTheme, screen, userEvent } from '@xlr-ui/tests/unit/testing-utils';
import { InputWithSend, InputWithSendProps } from '@xlr-ui/app/react/components/input-with-send/input-with-send.component';

describe('InputWithSend', () => {
    const onSendClick = jest.fn();
    const onStopClick = jest.fn();
    const placeholder = 'Type a message...';

    const defaultProps: InputWithSendProps = {
        ai: false,
        autoFocus: false,
        className: 'className',
        helperText: 'helperText',
        id: 'id',
        isSending: false,
        name: 'name',
        onSendClick,
        onStopClick,
        placeholder,
    };
    const render = (props = defaultProps) => {
        renderWithStoreAndTheme(<InputWithSend {...props} />, jest.fn(), {});
    };

    const getInput = () => screen.getByPlaceholderText(placeholder);
    const getStopButton = () => screen.getByTestId('input-with-send-cancel-button');
    const clickSendButton = async () => userEvent.click(screen.getByTestId('input-with-send-button'));
    const type = async (text: string) => await userEvent.type(getInput(), text);

    it('should render by default with multiline', () => {
        render();
        expect(getInput().tagName).toBe('TEXTAREA');
    });

    it('should render input if multiline is explicitly false', () => {
        render({ ...defaultProps, multiline: false });
        expect(getInput().tagName).toBe('INPUT');
    });

    it('should submit the message when clicking the send button', async () => {
        render();
        await type('Test message');
        await clickSendButton();
        expect(onSendClick).toHaveBeenCalledWith('Test message');
    });

    it('should submit the message by pressing enter', async () => {
        render();
        await type('Second message{Enter}');
        expect(onSendClick).toHaveBeenCalledWith('Second message');
    });

    it('should clean the message after sending', async () => {
        render();
        await type('Third message');
        expect(getInput()).toHaveValue('Third message');
        await type('{Enter}');
        expect(onSendClick).toHaveBeenCalledWith('Third message');
        expect(getInput()).toHaveValue('');
    });

    it('should not send the message if is empty', async () => {
        render();
        onSendClick.mockReset();
        await type('{Enter}');
        expect(onSendClick).not.toHaveBeenCalled();
    });

    it('should not send the message if is full of empty spaces', async () => {
        render();
        onSendClick.mockReset();
        await type('          {Enter}');
        expect(onSendClick).not.toHaveBeenCalled();
    });

    it('should disable input and prevent sending when disabled', async () => {
        onSendClick.mockReset();
        render({ ...defaultProps, disabled: true });

        const input = getInput();
        expect(input).toBeDisabled();

        await type('Should not type');
        expect(input).toHaveValue('');

        const sendButton = screen.getByRole('button');
        expect(sendButton).toBeDisabled();

        await userEvent.type(input, '{Enter}');
        expect(onSendClick).not.toHaveBeenCalled();
    });

    it('should show stop icon when sending', () => {
        render({ ...defaultProps, isSending: true });
        expect(getStopButton()).toBeVisible();
    });

    it('should call onStop when stop button is pressed', async () => {
        render({ ...defaultProps, isSending: true });
        await userEvent.click(getStopButton());
        expect(onStopClick).toHaveBeenCalled();
    });

    it('should disable input when isSending is true', () => {
        render({ ...defaultProps, isSending: true });
        expect(getInput()).toBeDisabled();
    });
});
