import { useEffect, useState } from 'react';
import isEmpty from 'lodash/isEmpty';
import remarkGfm from 'remark-gfm';
import remarkBreaks from 'remark-breaks';
import rehypeSanitize from 'rehype-sanitize';
import rehypeStringify from 'rehype-stringify';
import remarkParse from 'remark-parse';
import remarkRehype from 'remark-rehype';
import { unified } from 'unified';
import { variablesInterpolator } from '@xlr-ui/app/js/variables/variables-interpolator';
import { interpolateLinks, interpolateUsers } from '../helper';
import { ReleaseVariable, User } from '@xlr-ui/app/types';

export const MAX_LINES = 300;

export interface UseMarkdownParserProps {
    hasMentionsEnabled?: boolean;
    hasVariablesEnabled?: boolean;
    markdownText: string;
    releaseVariables?: ReleaseVariable;
    users?: User[];
}

export const useMarkdownParser = ({
    hasVariablesEnabled = false,
    hasMentionsEnabled = false,
    markdownText,
    releaseVariables,
    users,
}: UseMarkdownParserProps): string => {
    // states for html
    const [parsedHtml, setParsedHtml] = useState<string>('');

    useEffect(() => {
        let isMounted = true;
        if (!markdownText) return;
        const numberOfLines = (markdownText.match(/\n/g) || '').length + 1;
        const markdownParser = unified().use(remarkParse).use(remarkGfm);
        if (numberOfLines < MAX_LINES) {
            markdownParser.use(remarkBreaks);
        }
        markdownParser
            .use(remarkRehype)
            .use(rehypeSanitize)
            .use(rehypeStringify)
            .process(markdownText)
            .then((html) => {
                if (!isMounted) return;
                let htmlReplaced = html.value as string;
                if (hasVariablesEnabled) {
                    htmlReplaced = variablesInterpolator().interpolateInHtml(releaseVariables ?? {}, html.value as string) ?? htmlReplaced;
                }
                if (hasMentionsEnabled && users && !isEmpty(users)) {
                    htmlReplaced = interpolateUsers(htmlReplaced, users);
                }
                htmlReplaced = interpolateLinks(htmlReplaced);
                setParsedHtml(htmlReplaced);
            });

        // Ensures the component does not update after unmounting, preventing the memory leak
        return () => {
            isMounted = false;
        };
    }, [markdownText]);

    return parsedHtml;
};
