import { all, call, put, select, takeLatest } from 'redux-saga/effects';
import { PayloadAction } from '@reduxjs/toolkit';
import { SagaIterator } from 'redux-saga';
import { httpPOST } from '@xlr-ui/app/features/common/services/http';
import { DRAWER_HISTORY_ITEMS_PER_PAGE } from '../constants';
import { drawerHistory, DrawerHistoryState, getDrawerHistoryState, HistoryLogsFilter, LoadFilteredAndPagedHistoryLogsAction } from './drawer-history.reducer';
import { taskDrawer } from '@xlr-ui/app/features/tasks/ducks/task-drawer.reducer';
import { bootstrap } from '@xlr-ui/app/features/bootstrap/ducks/bootstrap-reducer';
import { checkIfLastPage, getFilterSettingsForLoadPayload } from '../../../helper';
import { ActivityLog } from '../../../types';

const { init, setLogs, setLogsFilter, setIsLoading, setIsLastPage, setPage, loadFilteredAndPagedHistoryLogs } = drawerHistory.actions;
const { addRail } = taskDrawer.actions;
const { afterLogin } = bootstrap.actions;

export function* initSaga(action: PayloadAction<HistoryLogsFilter>): SagaIterator {
    try {
        const page = 0;
        const logsFilter = action.payload;
        yield put(setIsLoading(true));
        yield put(setPage(page));
        const logs = yield call(loadLogs, page, logsFilter);
        const isLastPage = checkIfLastPage(logs, DRAWER_HISTORY_ITEMS_PER_PAGE);
        yield put(setIsLastPage(isLastPage));
        yield put(setLogs(logs));
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* loadFilteredAndPagedHistoryLogsSaga(action: PayloadAction<LoadFilteredAndPagedHistoryLogsAction>) {
    try {
        const { logsFilter, page } = action.payload;
        const { logs: oldLogs }: DrawerHistoryState = yield select(getDrawerHistoryState);
        yield put(setIsLoading(true));
        if (page === 0 && oldLogs.length > 0) {
            yield put(setLogs([]));
        }
        yield put(setPage(page));
        yield put(setLogsFilter(logsFilter));
        const logs: ActivityLog[] = yield call(loadLogs, page, logsFilter);
        let newLogs: ActivityLog[] = oldLogs;

        if (page === 0) {
            newLogs = logs;
        } else if (logs.length > 0) {
            newLogs = oldLogs.concat(logs);
        }

        const isLastPage = checkIfLastPage(logs, DRAWER_HISTORY_ITEMS_PER_PAGE);
        yield put(setIsLastPage(isLastPage));
        yield put(setLogs(newLogs));
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* loadLogs(page: number, logsFilter: HistoryLogsFilter) {
    const { containerId } = logsFilter;
    const { data: logs } = yield call(
        httpPOST,
        `activity/${containerId}/logs/search?page=${page}&resultsPerPage=${DRAWER_HISTORY_ITEMS_PER_PAGE}`,
        getFilterSettingsForLoadPayload(logsFilter.filterSettings),
    );
    return logs;
}

export function* addHistoryRailSaga() {
    yield put(addRail({ title: 'History', iconId: 'history', index: 6 }));
}

export default function* drawerHistorySaga() {
    yield all([
        takeLatest(init, initSaga),
        takeLatest(afterLogin, addHistoryRailSaga),
        takeLatest(loadFilteredAndPagedHistoryLogs, loadFilteredAndPagedHistoryLogsSaga),
    ]);
}
