import time
import urllib
from base64 import urlsafe_b64decode

import com.xhaus.jyson.JysonCodec as json

from xlrelease.HttpRequest import HttpRequest
from com.xebialabs.xlrelease.plugins.jenkins import JwtSigner


class OidcHelper:

    def __init__(self, jenkins_server, oidc_token=None, access_token=None):
        self.jenkins_server = jenkins_server
        self.issuer_url = jenkins_server['oidcIssuerUrl']
        self.client_id_or_email = jenkins_server['oidcClientIdOrEmail']
        self.client_secret = jenkins_server['oidcClientSecret']
        self.refresh_token = jenkins_server['oidcRefreshToken']
        self.private_key = jenkins_server['oidcPrivateKey']
        self.private_key_id = jenkins_server['oidcPrivateKeyId']
        self.scope = jenkins_server['oidcScope'] or 'openid email profile'
        self.oidc_token = oidc_token
        self.access_token = access_token
        self.validate_input_parameters()

        if (not self.oidc_token or not self.is_token_valid(self.oidc_token)) and (not self.access_token or not self.is_token_valid(self.access_token)):
            logger.debug("OIDC token is not valid or not provided, extracting a new token")
            self.extract_token()

    def validate_input_parameters(self):
        """
        Validates the input combinations required for OIDC token extraction.
        Raises an exception if the required combinations are not met.
        """
        if not self.issuer_url:
            raise ValueError("OIDC issuer URL is required")

        valid_combinations = [
            self.refresh_token and self.client_id_or_email and self.client_secret,
            self.client_id_or_email and self.private_key,
            self.client_id_or_email and self.client_secret
        ]

        if not any(valid_combinations):
            raise ValueError("Insufficient input parameters for token generation")

    def set_token_headers(self, request_headers):
        if self.oidc_token and self.jenkins_server['oidcTokenHeaderName']:
            request_headers[self.jenkins_server['oidcTokenHeaderName']] = "Bearer %s" % self.oidc_token
        if self.access_token and self.jenkins_server['accessTokenHeaderName']:
            request_headers[self.jenkins_server['accessTokenHeaderName']] = "Bearer %s" % self.access_token
        return self
    
    def get_http_request_parameters(self, url):
        params = {
            'url': url,
            'proxyHost': self.jenkins_server.get('proxyHost'),
            'proxyPort': self.jenkins_server.get('proxyPort'),
            'proxyUsername': self.jenkins_server.get('proxyUsername'),
            'proxyPassword': self.jenkins_server.get('proxyPassword'),
            'proxyDomain': self.jenkins_server.get('proxyDomain')}
        return params

    def extract_token(self):
        """
        Extracts an OIDC token by interacting with the OpenID Connect provider.

        This method retrieves the discovery document, determines supported grant types
        and authentication methods, and selects the appropriate grant type based on
        the available input credentials (refresh token, private key, or client secret).

        Supported grant types, listed by priority:
        - Refresh Token Grant (if refresh token is provided)
        - JWT Bearer Grant (if private key is provided and supported)
        - Client Credentials Grant (if private key is used with client_assertion or client secret is provided)

        Raises:
            Exception: If no supported token grant type is found or if the discovery document retrieval fails.
        """
        request_headers = {"Accept": "application/json", "Content-Type": "application/json"}

        #Fetch OIDC discovery document
        well_known_url = "{}/.well-known/openid-configuration".format(self.issuer_url)
        response = HttpRequest(self.get_http_request_parameters(well_known_url)).get(None, headers = request_headers)
        self.raise_for_status(response, "Failed to retrieve OpenID Connect discovery document")

        discovery = json.loads(response.getResponse())
        self.token_url = discovery["token_endpoint"]

        # Determine supported auth methods and grant types from discovery metadata
        grant_types = discovery.get("grant_types_supported", [])
        auth_methods = discovery.get("token_endpoint_auth_methods_supported", [])

        supports_jwt_grant = "urn:ietf:params:oauth:grant-type:jwt-bearer" in grant_types
        supports_client_credentials = "client_credentials" in grant_types
        supports_private_key_jwt = "private_key_jwt" in auth_methods

        # Determine support signing algorithm for assertion
        alg_values_supported = discovery.get("token_endpoint_auth_signing_alg_values_supported", [])
        self.signing_alg = "RS256"  
        if "RS256" not in alg_values_supported and alg_values_supported:
            self.signing_alg = alg_values_supported[0]

        logger.debug("supports_jwt_grant: %s, supports_client_credentials: %s, supports_private_key_jwt: %s, signing_alg: %s", supports_jwt_grant, supports_client_credentials, supports_private_key_jwt, self.signing_alg)

        if self.refresh_token:
            logger.debug("Using Refresh Token Grant (RFC 6749)")
            return self.use_refresh_token_grant()
                    
        elif self.private_key:
            if supports_jwt_grant:
                logger.debug("Using JWT Bearer Grant (RFC 7523)")
                return self.use_jwt_grant()
            else:
                logger.debug("Using Client Credentials Grant with private_key_jwt client authentication")
                if not supports_private_key_jwt:
                    logger.debug("Warning: Discovery document does not list private_key_jwt under supported auth methods")
                return self.use_client_credentials()
        
        elif self.client_secret: 
            logger.debug("Using Client Credentials Grant with client_secret authentication")
            if not supports_client_credentials:
                logger.debug("Warning: Discovery document does not list client_credentials under supported grant types")
            return self.use_client_credentials()
        
        else:
            raise Exception("No supported token grant type found for automation.")

    def use_refresh_token_grant(self):
        request_data = urllib.urlencode({
            "grant_type":    "refresh_token",
            "client_id":     self.client_id_or_email,
            "client_secret": self.client_secret,
            "refresh_token": self.refresh_token})
           
        oidc_request_headers={"Accept": "application/json", "Content-Type": "application/x-www-form-urlencoded"}

        response = HttpRequest(self.get_http_request_parameters(self.token_url)).post(None, request_data, headers=oidc_request_headers)
        if response.status != 200:
            raise Exception('Error generating OIDC token %s:%s' % (response.getStatus(), response.getResponse()))
            
        data=json.loads(response.getResponse())
        if "id_token" not in data and self.jenkins_server['oidcTokenHeaderName']:
            raise Exception('No ID token was returned. Ensure OpenID scope is included')

        self.oidc_token = data["id_token"]
        self.access_token =  data.get("access_token", None)
        

    # Retrieves an OIDC token using the JWT Bearer Grant Type
    def use_jwt_grant(self):
        signed_jwt = JwtSigner.signJwt(self.signing_alg, self.private_key, self.private_key_id, self.token_url, self.client_id_or_email, self.client_id_or_email)

        request_data = {
            "grant_type": "urn:ietf:params:oauth:grant-type:jwt-bearer",
            "assertion": signed_jwt
        }

        encoded_request_data = json.dumps(request_data)
        oidc_request_headers={"Accept": "application/json", "Content-Type": "application/json"}

        response = HttpRequest(self.get_http_request_parameters(self.token_url)).post(None, encoded_request_data, headers=oidc_request_headers)
        self.raise_for_status(response, "Failed to retrieve OIDC token using JWT grant")

        token_response = json.loads(response.getResponse())
        self.oidc_token = token_response.get("id_token", None)
        self.access_token =  token_response.get("access_token", None)
       
    
    # Retrieves an OIDC token using Client Credentials Grant Type
    def use_client_credentials(self):
        request_data = {"grant_type": "client_credentials", "scope": self.scope}
        
        if self.private_key:
            client_assertion = JwtSigner.signJwt(self.signing_alg, self.private_key, self.private_key_id, self.token_url, self.client_id_or_email, self.client_id_or_email)
            request_data["client_assertion_type"] = "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"
            request_data["client_assertion"] = client_assertion
        else:    
            request_data["client_id"] = self.client_id_or_email
            request_data["client_secret"] = self.client_secret

        encoded_request_data = urllib.urlencode(request_data)
        oidc_request_headers = {"Accept": "application/json", "Content-Type": "application/x-www-form-urlencoded"}

        response = HttpRequest(self.get_http_request_parameters(self.token_url)).post(None, encoded_request_data, headers=oidc_request_headers)
        self.raise_for_status(response, "Failed to retrieve OIDC token using Client Credentials Grant")
        token_response = json.loads(response.getResponse())

        self.oidc_token = token_response.get("id_token", None)
        self.access_token = token_response.get("access_token", None)

    def raise_for_status(self, response, message="Unexpected error"):
        if response.status != 200:
            raise Exception("{}: {}".format(message, response.getResponse()))

    # Decodes a JWT (JSON Web Token) without verifying its signature
    def decode_jwt_no_verify(self, token):
        parts = token.split(".")
        if len(parts) != 3:
            raise ValueError("Invalid JWT format")

        payload_b64 = parts[1]
        payload_b64 += '=' * (-len(payload_b64) % 4)
        if isinstance(payload_b64, unicode):
            payload_b64 = payload_b64.encode('utf-8')

        payload_json = urlsafe_b64decode(payload_b64)
        return json.loads(payload_json)

    # Check if the token is valid for a minimum 120 seconds
    def is_token_valid(self, token, min_valid_seconds=120):
        try:
            payload = self.decode_jwt_no_verify(token)
            exp_ts = payload.get("exp")
            if exp_ts is None:
                return False

            now = time.time()
            return (exp_ts - now) > min_valid_seconds
        except Exception as e:
            return False