import React, { ChangeEvent } from 'react';
import { ReactWrapper } from 'enzyme';
import { DotTable, DotTablePagination } from '@digital-ai/dot-components';
import { mountComponentWithStore } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ExternalDeploymentsTable } from './external-deployments-table.component';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialPage, initialState } from '../ducks/external-deployments.reducer';
import { mockApplicationsMap, mockEnvironmentsMap, mockLiveDeploymentData, mockServersMap, pluginInfoDeploy } from '../__mocks__/external-deployments.mocks';
import { Search } from '../../../../../../../../../core/xlr-ui/app/react/components/search/search';
import { STATUS_HTTP_CONNECTION } from '../constants';

const { getFilters, loadExternalDeployments, setCondition, storeFilters, setPage, subscribeToSseStream, unsubscribeFromSseStream, getAvailablePlugins } =
    folderExternalDeployments.actions;
const dispatch = jest.fn();
describe('ExternalDeploymentsTable component', () => {
    let wrapper: ReactWrapper;
    const onViewChange = jest.fn();
    const defaultProps = {
        folder: { id: 'folderId', title: 'Folder' },
        onViewChange,
    };

    const mountComponent = (state: FolderExternalDeploymentsState = initialState) => {
        wrapper = mountComponentWithStore(<ExternalDeploymentsTable {...defaultProps} />, dispatch, {
            folderExternalDeployments: state,
        });
        expect(dispatch).toBeCalledWith(getFilters('folderId'));
        expect(dispatch).toBeCalledWith(loadExternalDeployments());
        expect(dispatch).toBeCalledWith(subscribeToSseStream());
        expect(dispatch).toBeCalledWith(storeFilters());
        expect(dispatch).toBeCalledWith(getAvailablePlugins(STATUS_HTTP_CONNECTION));
    };
    beforeEach(() => {
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
        expect(dispatch).toBeCalledWith(unsubscribeFromSseStream());
    });
    const searchTable = (selector: string) => wrapper.find(`.external-deployments-table ${selector}`);

    it('should show the title, search box and table heading with no results', () => {
        expect(searchTable('h1.dot-typography').at(0).text()).toStrictEqual('Application deployments (0)');
        expect(searchTable('.search input').props().placeholder).toStrictEqual('Type to filter');
        expect(searchTable('p.dot-typography').at(0).text()).toStrictEqual('Application name');
        expect(searchTable('p.dot-typography').at(1).text()).toStrictEqual('Environment');
        expect(searchTable('p.dot-typography').at(2).text()).toStrictEqual('Version');
        expect(searchTable('p.dot-typography').at(3).text()).toStrictEqual('Status');
        expect(searchTable('p.dot-typography').at(4).text()).toStrictEqual('Updated');
    });

    it('should trigger filter search', () => {
        jest.useFakeTimers();
        jest.spyOn(global, 'clearTimeout');
        jest.spyOn(global, 'setTimeout');
        const e1 = { target: { value: 'filter' } };
        wrapper.find(Search).invoke('onFilterChange')?.(e1 as unknown as ChangeEvent<HTMLInputElement>);
        wrapper.update();
        expect(setTimeout).toHaveBeenCalledTimes(1);
        expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), 500);
        jest.runAllTimers();
        expect(dispatch).toHaveBeenCalledWith(setCondition('filter'));
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
        const e2 = { target: { value: 'filter 2' } };
        wrapper.find(Search).invoke('onFilterChange')?.(e2 as unknown as ChangeEvent<HTMLInputElement>);
        wrapper.update();
        expect(clearTimeout).toHaveBeenCalledTimes(1);
        jest.runAllTimers();
        expect(dispatch).toHaveBeenCalledWith(setCondition('filter 2'));
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
    });

    it('should not show one row and trigger page change', () => {
        mountComponent({
            ...initialState,
            applications: mockApplicationsMap,
            count: 1,
            environments: mockEnvironmentsMap,
            connectionServers: mockServersMap,
            liveDeployments: [mockLiveDeploymentData],
        });
        expect(searchTable('.dot-typography').at(0).text()).toStrictEqual('Application deployments (1)');
        expect(searchTable('.external-deployment-card').exists()).toBeFalsy();
        wrapper.find(DotTable).invoke('onUpdateData')?.('asc', 'column', 1, 10);
        expect(dispatch).toHaveBeenCalledWith(
            setPage({
                ...initialPage,
                folderId: defaultProps.folder.id,
                order: 'asc',
                orderBy: 'column',
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
    });

    it('should show one row and change page and results per page', () => {
        mountComponent({
            ...initialState,
            applications: mockApplicationsMap,
            connectionServers: mockServersMap,
            count: 1,
            environments: mockEnvironmentsMap,
            liveDeployments: [mockLiveDeploymentData],
            validServerCards: [pluginInfoDeploy],
        });
        expect(searchTable('.dot-typography').at(0).text()).toStrictEqual('Application deployments (1)');
        expect(searchTable('.external-deployment-card').exists()).toBeTruthy();
        wrapper.find(DotTablePagination).invoke('onPageChange')?.(2);
        expect(dispatch).toHaveBeenCalledWith(
            setPage({
                ...initialPage,
                folderId: defaultProps.folder.id,
                page: 2,
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
        wrapper.find(DotTablePagination).invoke('onRowsPerPageChange')?.({ target: { value: 10 } } as never);
        expect(dispatch).toHaveBeenCalledWith(
            setPage({
                ...initialPage,
                folderId: defaultProps.folder.id,
                resultsPerPage: 10,
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
    });

    it('should show connection servers error', () => {
        mountComponent({
            ...initialState,
            connectionErrors: ['Connection error 1', 'Connection error 2'],
        });
        expect(searchTable('.external-deployments-connection-errors').exists()).toBeTruthy();
        expect(searchTable('.external-deployments-connection-errors .dot-typography').at(0).text()).toStrictEqual('2 disconnected servers');
    });

    it('should show live deployment configuration table', () => {
        mountComponent({
            ...initialState,
            isLiveDeploymentConfigurationOpen: true,
        });
        expect(searchTable('.external-deployments-connection-errors').exists()).toBeFalsy();
        expect(searchTable('.external-deployment-card').exists()).toBeFalsy();
        expect(wrapper.find('.live-deployment-configuration-table').exists()).toBeTruthy();
    });
});
