import { all, call, put, select, takeEvery, takeLatest } from 'redux-saga/effects';
import moment from 'moment';
import { IHttpResponse } from 'angular';
import { analytics, AnalyticsState, CardSearch, FavoriteDashboardAction, getAnalyticsState } from './analytics.reducer';
import { PayloadAction } from '@reduxjs/toolkit';
import { ANALYTICS_CATEGORIES, ANALYTICS_PLATFORM_AUTHORS, INTELLIGENCE_SERVER_TITLE } from '../constants';
import { DEFAULT_PAGINATION } from '../../../../../../../../../core/xlr-ui/app/constants/pagination';
import { FilterQueryParams } from '../../../../../../../../../core/xlr-ui/app/types';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import {
    convertArrayToString,
    createAnalyticsApiUrl,
    extractRegionFromUrl,
    getFilteredSampleCards,
    mapAuthorName,
    mapSupersetToAnalyticsCard,
    mapSupersetToAnalyticsCards,
} from '../helpers';
import { ConfigurationService } from '../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import { SearchConfigurationResponse } from '../../../../../../../../../core/xlr-ui/app/features/configuration/types';
import { httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import {
    AccessTokenResponse,
    AnalyticsCardData,
    AnalyticsClientSettings,
    DashboardGetResponse,
    DashboardListResponse,
    DashboardMetadataResponse,
    IntelligenceHttpConnection,
    PlatformAuthor,
    PlatformListDashboardWithPaginationResponse,
    PlatformUserResponse,
} from '../types';
import ToastrFactory from '../../../../../../../../../core/xlr-ui/app/js/util/toastrFactory';

export const toaster = ToastrFactory();

const {
    addToUpdatingDashboardIds,
    favoriteDashboardFromDetails,
    favoriteDashboardFromList,
    filterCards,
    init,
    loadDashboard,
    loadFavoriteCards,
    removeFromFavoritesCards,
    removeFromUpdatingDashboardIds,
    replaceCardInAllCards,
    setAllCards,
    setAuthors,
    setCardSearch,
    setCategories,
    setFavoriteCards,
    setIntelligenceConfiguration,
    setIsCategoriesLoading,
    setIsLoading,
    setManageDashboardUrl,
    setSelectedDashboard,
    setTokenData,
    setTotalAvailableCards,
    setTotalAvailableFavoriteCards,
} = analytics.actions;

export type AnalyticsTabData = [AnalyticsCardData[], number];
export type AnalyticsCombinedTabsData = [AnalyticsTabData, AnalyticsTabData];

export function* initSaga(action: PayloadAction<Partial<CardSearch> | undefined>) {
    const payloadSearch = action.payload;
    const cardSearch = {
        ...payloadSearch,
        isFavorite: false,
        page: payloadSearch?.page ? payloadSearch?.page : DEFAULT_PAGINATION.page,
        itemsPerPage: payloadSearch?.itemsPerPage ? payloadSearch?.itemsPerPage : DEFAULT_PAGINATION.itemsPerPage,
    };

    yield put(setIsLoading(true));
    yield put(setAllCards([]));
    yield call(checkIfIntelligenceIsConfigured);
    yield call(loadMetadata);
    yield call(setFilters, cardSearch);

    const [[allCards, totalAvailableCards], [favoriteCards, totalAvailableFavoriteCards]]: AnalyticsCombinedTabsData = yield all([
        call(loadAnalyticsCardsData),
        call(loadAnalyticsCardsData, true),
    ]);

    yield put(setIsLoading(false));
    yield put(setFavoriteCards(favoriteCards));
    yield put(setTotalAvailableFavoriteCards(totalAvailableFavoriteCards));
    yield put(setAllCards(allCards));
    yield put(setTotalAvailableCards(totalAvailableCards));
    yield call(setManageDashboardUrlIfUserHasRole);
}

export function* loadAnalyticsCardsForBothTabs() {
    const [[allCards, totalAvailableCards], [favoriteCards, totalAvailableFavoriteCards]]: AnalyticsCombinedTabsData = yield all([
        call(loadAnalyticsCardsData),
        call(loadAnalyticsCardsData, true),
    ]);
    yield put(setFavoriteCards(favoriteCards));
    yield put(setTotalAvailableFavoriteCards(totalAvailableFavoriteCards));
    yield put(setAllCards(allCards));
    yield put(setTotalAvailableCards(totalAvailableCards));
}

export function* setManageDashboardUrlIfUserHasRole() {
    const { intelligenceConfiguration }: AnalyticsState = yield select(getAnalyticsState);
    if (intelligenceConfiguration) {
        const { entity: user }: PlatformUserResponse = yield call(
            doPostWithToken<PlatformUserResponse>,
            `api/extension/intelligence/user?serverId=${intelligenceConfiguration.id}`,
        );
        if (user.roles.includes('account-analytics-author')) {
            const serverUrl = intelligenceConfiguration.url;
            const vanity = user.account.vanity_domain;
            const region = extractRegionFromUrl(serverUrl);
            if (region) {
                yield put(setManageDashboardUrl(`https://${vanity}.${region}.digital.ai/accounts/${user.account_id}/analytics/dashboards?app=RELEASE`));
            }
        }
    }
}

export function* checkIfIntelligenceIsConfigured() {
    const configurationService: ConfigurationService = yield call(getAngularService, 'ConfigurationService');
    const intelligenceServerResponse: SearchConfigurationResponse = yield call(
        [configurationService, configurationService.searchConfiguration],
        'dai.IntelligenceServer',
        INTELLIGENCE_SERVER_TITLE,
    );
    const intelligenceServer = intelligenceServerResponse['dai.IntelligenceServer'];
    if (intelligenceServer && intelligenceServer.length) {
        yield put(setIntelligenceConfiguration(intelligenceServer[0] as IntelligenceHttpConnection));
    } else {
        yield put(setIntelligenceConfiguration(undefined));
    }
}

export function* loadMetadata() {
    const { intelligenceConfiguration }: AnalyticsState = yield select(getAnalyticsState);
    let dashboardCategories: string[] = ANALYTICS_CATEGORIES;
    let dashboardAuthors: PlatformAuthor[] = ANALYTICS_PLATFORM_AUTHORS;
    yield put(setIsCategoriesLoading(true));
    if (intelligenceConfiguration) {
        const { entity }: DashboardMetadataResponse = yield call(
            doPostWithToken<DashboardMetadataResponse>,
            `api/extension/intelligence/metadata?serverId=${intelligenceConfiguration.id}`,
        );
        dashboardCategories = entity.categories.sort();
        dashboardAuthors = entity.authors.sort((a, b) => {
            if (a.full_name < b.full_name) return -1;
            if (a.full_name > b.full_name) return 1;
            return 0;
        });
    }
    yield put(setCategories(dashboardCategories));
    yield put(setAuthors(mapAuthorName(dashboardAuthors)));
    yield put(setIsCategoriesLoading(false));
}

export function* fetchDashboardsFromApi(intelligenceConfigurationId: string, cardSearch: CardSearch) {
    const url = createAnalyticsApiUrl(cardSearch, intelligenceConfigurationId);
    const { entity }: DashboardListResponse = yield call(doPostWithToken<DashboardListResponse>, url);
    return entity;
}

export function* loadAnalyticsCardsData(isFavorites = false, startPage?: number) {
    const { intelligenceConfiguration, cardSearch }: AnalyticsState = yield select(getAnalyticsState);
    let cards = isFavorites ? [] : getFilteredSampleCards(cardSearch);
    let totalAvailableCards = cards.length;
    if (intelligenceConfiguration) {
        const entity: PlatformListDashboardWithPaginationResponse = yield call(fetchDashboardsFromApi, intelligenceConfiguration.id, {
            ...cardSearch,
            page: startPage ?? cardSearch.page,
            isFavorite: isFavorites,
        });
        cards = mapSupersetToAnalyticsCards(entity.results, intelligenceConfiguration?.folderColumnName);
        totalAvailableCards = entity.pagination.total_available;
    }
    return [cards, totalAvailableCards];
}

export function* favoriteDashboardFromListSaga(action: PayloadAction<FavoriteDashboardAction>) {
    const { intelligenceConfiguration }: AnalyticsState = yield select(getAnalyticsState);
    const { id, isUnfavorite } = action.payload;

    if (intelligenceConfiguration) {
        yield put(addToUpdatingDashboardIds(id));
        const updatedCard: AnalyticsCardData = yield call(sendFavoriteDashboardRequest, intelligenceConfiguration, id, isUnfavorite);

        if (isUnfavorite) {
            yield put(removeFromFavoritesCards(id));
        } else {
            yield put(loadFavoriteCards());
        }
        yield put(replaceCardInAllCards(updatedCard));
        yield put(removeFromUpdatingDashboardIds(id));
    }
}

export function* favoriteDashboardFromDetailsSaga(action: PayloadAction<FavoriteDashboardAction>) {
    const { intelligenceConfiguration }: AnalyticsState = yield select(getAnalyticsState);
    const { id, isUnfavorite } = action.payload;

    if (intelligenceConfiguration) {
        yield put(addToUpdatingDashboardIds(id));
        const updatedCard: AnalyticsCardData = yield call(sendFavoriteDashboardRequest, intelligenceConfiguration, id, isUnfavorite);
        yield put(setSelectedDashboard(updatedCard));
        yield put(removeFromUpdatingDashboardIds(id));
    }
}

export function* sendFavoriteDashboardRequest(intelligenceConfiguration: IntelligenceHttpConnection, dashboardId: string, isUnfavorite: boolean) {
    let url = `api/extension/intelligence/dashboards/favorite?serverId=${intelligenceConfiguration.id}&dashboardId=${dashboardId}`;
    if (isUnfavorite) url += '&unfavorite=1';
    const dashboardResponse: DashboardGetResponse = yield call(doPostWithToken<DashboardGetResponse>, url);
    return mapSupersetToAnalyticsCard(dashboardResponse.entity, intelligenceConfiguration?.folderColumnName);
}

export function* loadFavoriteCardsSaga() {
    const [favoriteCards, totalAvailableFavoriteCards]: AnalyticsTabData = yield call(loadAnalyticsCardsData, true, 0);
    yield put(setFavoriteCards(favoriteCards));
    yield put(setTotalAvailableFavoriteCards(totalAvailableFavoriteCards));
}

export function* setFilters(cardSearch: CardSearch) {
    const { authors, categories }: AnalyticsState = yield select(getAnalyticsState);
    // if authors or categories in cardSearch contain non-existing options (e.g. when manually typing to url some value that is not present in filters drawer),
    // then remove those invalid filters from cardSearch
    const authorsSearch = cardSearch.authorNames?.filter((author) => authors.includes(author));
    const categoriesSearch = cardSearch.categories?.filter((category) => categories.includes(category));
    const cleanCardSearch = {
        ...cardSearch,
        authorNames: authorsSearch?.length ? authorsSearch : undefined,
        categories: categoriesSearch?.length ? categoriesSearch : undefined,
    };
    yield put(setCardSearch(cleanCardSearch));
}

export function* filterCardsSaga(action: PayloadAction<CardSearch>) {
    const { isFavorite, page } = action.payload;
    const { allCards, favoriteCards }: AnalyticsState = yield select(getAnalyticsState);
    yield call(setFilters, action.payload);
    yield put(setIsLoading(true));
    let newCards: AnalyticsCardData[];

    // When page is 0 that means that we are filtering, and when we filter we need to fetch ALL cards and favorites
    if (page === 0) {
        yield call(loadAnalyticsCardsForBothTabs);
    } else {
        const [cards, totalAvailableCards]: AnalyticsTabData = yield call(loadAnalyticsCardsData, isFavorite);
        newCards = isFavorite ? favoriteCards.concat(cards) : allCards.concat(cards);
        if (isFavorite) {
            yield put(setFavoriteCards(newCards));
            yield put(setTotalAvailableFavoriteCards(totalAvailableCards));
        } else {
            yield put(setAllCards(newCards));
            yield put(setTotalAvailableCards(totalAvailableCards));
        }
    }

    yield put(setIsLoading(false));
}

export function* setCardSearchSaga(action: PayloadAction<CardSearch | undefined>) {
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    const clientSettings: AnalyticsClientSettings = yield call(getAngularService, 'ClientSettings');
    const cardSearch = action.payload;
    const { page, itemsPerPage, ...cardSearchWithoutPagination } = cardSearch ?? {};
    clientSettings.setAnalyticsListFilters(cardSearch);
    FiltersQueryParams.update({
        ...cardSearchWithoutPagination,
        categories: convertArrayToString(cardSearch?.categories),
        authorNames: convertArrayToString(cardSearch?.authorNames),
    });
}

export function* loadDashboardSaga(action: PayloadAction<string>) {
    yield call(checkIfIntelligenceIsConfigured);
    const { allCards, intelligenceConfiguration }: AnalyticsState = yield select(getAnalyticsState);
    const dashboardId = action.payload;
    if (intelligenceConfiguration) {
        let dashboard;
        if (!allCards.length) {
            // this might happen if we just refresh the details page
            const { entity }: DashboardGetResponse = yield call(
                doPostWithToken<DashboardGetResponse>,
                `api/extension/intelligence/dashboard?serverId=${intelligenceConfiguration?.id}&dashboardId=${dashboardId}`,
            );
            dashboard = mapSupersetToAnalyticsCard(entity, intelligenceConfiguration.folderColumnName);
        } else {
            dashboard = allCards.find((c) => c.id === dashboardId);
        }
        if (dashboard) {
            yield put(setSelectedDashboard(dashboard));
        } else {
            yield call(toaster.error, `Dashboard [${dashboardId}] can't be found`);
        }
    }
}

export function* getToken() {
    const { tokenExpiresAt, tokenValue } = yield select(getAnalyticsState);
    let shouldFetchToken = true;
    if (tokenExpiresAt && tokenValue) {
        const expiresMoment = moment.utc(tokenExpiresAt * 1000);
        const oneMinuteInFutureMoment = moment().add(1, 'minute');
        shouldFetchToken = expiresMoment.isBefore(oneMinuteInFutureMoment);
    }
    if (!shouldFetchToken) {
        return tokenValue;
    }
    try {
        const {
            data: { tokenValue: token, expiresAt },
        }: IHttpResponse<AccessTokenResponse> = yield call(httpGET, 'oidc/token', true);
        yield put(setTokenData({ tokenExpiresAt: expiresAt, tokenValue: token }));
        return token;
    } catch (e) {
        yield call(toaster.error, 'Something went wrong while connecting to Analytics. Your session might be expired.');
        throw e;
    }
}

export function* doPostWithToken<T>(url: string) {
    const tokenValue: string = yield call(getToken);
    const response: IHttpResponse<T> = yield call(httpPOST, url, { accessToken: tokenValue });
    return response.data;
}

export function* analyticsSaga() {
    yield all([
        takeLatest(init, initSaga),
        takeLatest(filterCards, filterCardsSaga),
        takeEvery(favoriteDashboardFromDetails, favoriteDashboardFromDetailsSaga),
        takeEvery(favoriteDashboardFromList, favoriteDashboardFromListSaga),
        takeLatest(setCardSearch, setCardSearchSaga),
        takeLatest(loadFavoriteCards, loadFavoriteCardsSaga),
        takeLatest(loadDashboard, loadDashboardSaga),
    ]);
}
