import { SagaIterator } from 'redux-saga';
import { all, call, put, race, select, take, takeEvery, takeLatest } from 'redux-saga/effects';
import { applicationManagement } from './managed-application.reducer';
import {
    createOrUpdateVariable,
    executeCreateManagedApplicationsFromWebhookSourceAction,
    executeFetchManagedApplicationsAction,
    getFiltersAction,
    loadEditPermissionAction,
    managedApplicationSaga,
    navigateToWorkflowSaga,
    storeFiltersAction,
    toaster,
    withLoadingState,
} from './managed-application.saga';

import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { getApplicationManagementState } from './managed-application.selectors';
import { httpDELETE, httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { getReleaseVariables, updateReleaseVariable } from '../../../../../../../../../core/xlr-ui/app/features/tasks/ducks/variables-service.saga';
import { workflow } from '../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { ManagedApplicationsPage } from '../managed-application.types';
import { DELETE_BUTTON_ACTION, EDIT_BUTTON_ACTION } from '../components/created-applications-table/managed-application.component';
import { hasAnyOfPermissions } from '../../../../../../../../../core/xlr-ui/app/js/auth/authenticator.saga';
import { Permissions } from '../../../../../../../../../core/xlr-ui/app/js/security/permissions-constants';
import { Folder, Variable } from '../../../../../../../../../core/xlr-ui/app/types';

const {
    createManagedApplicationsFromWebhookSource,
    getFilters,
    loadManagedApplications,
    navigateToWorkflow,
    setCondition,
    setManagedApplications,
    setManagedApplicationsCount,
    setManagedApplicationsTotalCount,
    setPage,
    storeFilters,
    loadEditPermission,
    setEditPermission,
} = applicationManagement.actions;
const { createWorkflowError, createWorkflowSuccess, runWorkflow } = workflow.actions;

describe('applicationManagementSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = managedApplicationSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeEvery(createManagedApplicationsFromWebhookSource, executeCreateManagedApplicationsFromWebhookSourceAction),
                takeEvery(getFilters, getFiltersAction),
                takeLatest(loadManagedApplications, executeFetchManagedApplicationsAction),
                takeLatest(navigateToWorkflow, navigateToWorkflowSaga),
                takeEvery(storeFilters, storeFiltersAction),
                takeEvery(loadEditPermission, loadEditPermissionAction),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    const folder: Folder = {
        title: 'folder-1',
        id: 'folder-1',
        $metadata: {
            security: {
                teams: [],
                permissions: [Permissions.EDIT_APP_PIPELINES],
            },
        },
    };
    const condition = 'con';
    const page: ManagedApplicationsPage = {
        folderId: 'folder-1',
        order: 'asc',
        orderBy: 'environment',
        page: 0,
        resultsPerPage: 10,
    };
    const filters = {
        condition,
        ...page,
    };
    const setManagedApplicationsFilters = jest.fn();
    const getManagedApplicationsFilters = jest.fn(() => filters);
    const clientSettings = {
        getManagedApplicationsFilters,
        setManagedApplicationsFilters,
    };
    const update = jest.fn();
    const filtersQueryParams = {
        update,
    };

    const appIdVariable: Variable = {
        id: 'appIdVariable',
        key: 'managedApplicationId',
        requiresValue: false,
        showOnReleaseStart: false,
        type: 'xlrelease.StringVariable',
    };
    const appNameVariable: Variable = {
        id: 'appNameVariable',
        key: 'applicationName',
        requiresValue: false,
        showOnReleaseStart: false,
        type: 'xlrelease.StringVariable',
    };
    const envTagVariable: Variable = {
        id: 'envTagVariable',
        key: 'environmentTag',
        requiresValue: false,
        showOnReleaseStart: false,
        type: 'xlrelease.StringVariable',
    };
    const managedByVariable: Variable = {
        id: 'managedByVariable',
        key: 'managedBy',
        requiresValue: false,
        showOnReleaseStart: false,
        type: 'xlrelease.StringVariable',
    };
    const clusterUrlVariable: Variable = {
        id: 'clusterUrlVariable',
        key: 'clusterUrl',
        requiresValue: false,
        showOnReleaseStart: false,
        type: 'xlrelease.StringVariable',
    };
    const folderIdVariable: Variable = {
        id: 'folderIdVariable',
        key: 'folderId',
        requiresValue: false,
        showOnReleaseStart: false,
        type: 'xlrelease.StringVariable',
    };
    const deleteWorkflowIdVariable: Variable = {
        id: 'deleteWorkflowIdVariable',
        key: 'deleteWorkflowId',
        requiresValue: false,
        showOnReleaseStart: false,
        type: 'xlrelease.StringVariable',
    };
    const updateWorkflowIdVariable: Variable = {
        id: 'updateWorkflowIdVariable',
        key: 'updateWorkflowId',
        requiresValue: false,
        showOnReleaseStart: false,
        type: 'xlrelease.StringVariable',
    };
    const variables: Array<Variable> = [
        appIdVariable,
        appNameVariable,
        envTagVariable,
        managedByVariable,
        clusterUrlVariable,
        folderIdVariable,
        deleteWorkflowIdVariable,
        updateWorkflowIdVariable,
    ];
    const managedApplication = {
        id: 'appId',
        applicationName: 'test',
        managedBy: 'Release',
        environmentTag: 'DEV',
        applicationReference: 'https://kubernetes.default.svc',
        folderId: 'folderId',
        deleteWorkflowId: 'deleteWorkflowId',
        updateWorkflowId: 'updateWorkflowId',
    };

    describe('storeFiltersAction', () => {
        it('should store filters to local storage', () => {
            const gen: SagaIterator = storeFiltersAction();
            expect(gen.next().value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page, condition }).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call(getAngularService, 'FiltersQueryParams'));
            expect(gen.next(filtersQueryParams).done).toBeTruthy();
            expect(update).toHaveBeenCalled();
        });
    });

    describe('executeCreateManagedApplicationsFromWebhookSourceAction', () => {
        it('should create managed applications from webhook source', () => {
            const webhookSourceId = 'Configuration/Custom/Configuration1682004056084295907f2e9c654f049b';
            const folderId = 'SomeFolder';

            const gen: SagaIterator = executeCreateManagedApplicationsFromWebhookSourceAction({ payload: { webhookSourceId, folderId }, type: 'any' });
            expect(gen.next().value).toEqual(
                call(withLoadingState, call(httpPOST, `/api/v1/managed-application/create-from-webhook-source/${webhookSourceId}?folderId=${folderId}`)),
            );
            expect(gen.next().value).toEqual(call(toaster.success, `Successfully created managed applications from discovered applications`));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeFetchManagedApplicationsAction', () => {
        it('should fetch managed apps', () => {
            const gen: SagaIterator = executeFetchManagedApplicationsAction();
            expect(gen.next().value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page, condition }).value).toStrictEqual(
                call(
                    withLoadingState,
                    call(
                        httpGET,
                        `api/v1/managed-application?page=${page.page}&folderId=${page.folderId}&resultsPerPage=${
                            page.resultsPerPage
                        }&condition=${encodeURIComponent(condition)}`,
                        true,
                    ),
                ),
            );
            expect(gen.next({ data: { count: 2, managedApplications: [], totalCount: 200 } }).value).toStrictEqual(put(setManagedApplications([])));
            expect(gen.next().value).toStrictEqual(put(setManagedApplicationsCount(2)));
            expect(gen.next().value).toStrictEqual(put(setManagedApplicationsTotalCount(200)));
            expect(gen.next().done).toBeTruthy();
        });
        it('should handle undefined condition', () => {
            const gen: SagaIterator = executeFetchManagedApplicationsAction();
            expect(gen.next().value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page }).value).toStrictEqual(
                call(
                    withLoadingState,
                    call(
                        httpGET,
                        `api/v1/managed-application?page=${page.page}&folderId=${page.folderId}&resultsPerPage=${page.resultsPerPage}&condition=`,
                        true,
                    ),
                ),
            );
            expect(gen.next({ data: { count: 2, managedApplications: [], totalCount: 200 } }).value).toStrictEqual(put(setManagedApplications([])));
            expect(gen.next().value).toStrictEqual(put(setManagedApplicationsCount(2)));
            expect(gen.next().value).toStrictEqual(put(setManagedApplicationsTotalCount(200)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('getFiltersAction', () => {
        it('should get filters from local storage and update state', () => {
            const gen: SagaIterator = getFiltersAction({ payload: 'folder-1', type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call(getAngularService, '$stateParams'));
            expect(gen.next(clientSettings).value).toStrictEqual(put(setCondition(condition)));
            expect(gen.next(clientSettings).value).toStrictEqual(put(setPage(page)));
            expect(gen.next(clientSettings).done).toBeTruthy();
            expect(getManagedApplicationsFilters).toHaveBeenCalled();
        });
        it('should handle empty filters', () => {
            const getEmptyManagedApplicationsFilters = jest.fn();
            const gen: SagaIterator = getFiltersAction({ payload: 'folder-1', type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next({ ...clientSettings, getManagedApplicationsFilters: getEmptyManagedApplicationsFilters }).value).toStrictEqual(
                call(getAngularService, '$stateParams'),
            );
            expect(gen.next({ getManagedApplicationsFilters: getEmptyManagedApplicationsFilters }).done).toBeTruthy();
            expect(getEmptyManagedApplicationsFilters).toHaveBeenCalled();
        });
    });

    describe('navigateToWorkflowSaga', () => {
        it('should fetch managed app then trigger update workflow with updated variable and cleanup', () => {
            const payload = {
                actionName: EDIT_BUTTON_ACTION,
                managedApplicationId: managedApplication.id,
                workflowId: managedApplication.updateWorkflowId,
            };
            const action = { payload, type: 'any' };
            const gen: SagaIterator = navigateToWorkflowSaga(action);
            expect(gen.next().value).toStrictEqual(call(httpGET, `api/v1/managed-application/${managedApplication.id}`));
            expect(gen.next({ data: managedApplication }).value).toStrictEqual(select(getApplicationManagementState));
            expect(gen.next({ page }).value).toStrictEqual(call(getReleaseVariables, payload.workflowId, true));
            expect(gen.next(variables).value).toStrictEqual(
                call(
                    createOrUpdateVariable,
                    managedApplication.updateWorkflowId,
                    variables,
                    appIdVariable.key,
                    managedApplication.id,
                    'xlrelease.StringVariable',
                ),
            );
            expect(gen.next(appIdVariable).value).toStrictEqual(
                call(
                    createOrUpdateVariable,
                    managedApplication.updateWorkflowId,
                    variables,
                    appNameVariable.key,
                    managedApplication.applicationName,
                    'xlrelease.StringVariable',
                ),
            );
            expect(gen.next(appNameVariable).value).toStrictEqual(
                call(
                    createOrUpdateVariable,
                    managedApplication.updateWorkflowId,
                    variables,
                    envTagVariable.key,
                    managedApplication.environmentTag,
                    'xlrelease.StringVariable',
                ),
            );
            expect(gen.next(envTagVariable).value).toStrictEqual(
                call(
                    createOrUpdateVariable,
                    managedApplication.updateWorkflowId,
                    variables,
                    managedByVariable.key,
                    managedApplication.managedBy,
                    'xlrelease.StringVariable',
                ),
            );
            expect(gen.next(managedByVariable).value).toStrictEqual(
                call(
                    createOrUpdateVariable,
                    managedApplication.updateWorkflowId,
                    variables,
                    clusterUrlVariable.key,
                    managedApplication.applicationReference,
                    'xlrelease.StringVariable',
                ),
            );
            expect(gen.next(clusterUrlVariable).value).toStrictEqual(
                call(
                    createOrUpdateVariable,
                    managedApplication.updateWorkflowId,
                    variables,
                    folderIdVariable.key,
                    managedApplication.folderId,
                    'xlrelease.StringVariable',
                ),
            );
            expect(gen.next(folderIdVariable).value).toStrictEqual(
                call(
                    createOrUpdateVariable,
                    managedApplication.updateWorkflowId,
                    variables,
                    deleteWorkflowIdVariable.key,
                    managedApplication.deleteWorkflowId,
                    'xlrelease.StringVariable',
                ),
            );
            expect(gen.next(deleteWorkflowIdVariable).value).toStrictEqual(
                call(
                    createOrUpdateVariable,
                    managedApplication.updateWorkflowId,
                    variables,
                    updateWorkflowIdVariable.key,
                    managedApplication.updateWorkflowId,
                    'xlrelease.StringVariable',
                ),
            );
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            expect(gen.next(updateWorkflowIdVariable).value).toStrictEqual(put(runWorkflow({ folderId: page.folderId, workflow: { id: payload.workflowId } })));
            expect(gen.next().value).toStrictEqual(
                race({
                    error: take(createWorkflowError.type),
                    success: take(createWorkflowSuccess.type),
                }),
            );

            expect(gen.next().value).toStrictEqual(all(variables.map((v) => call(updateReleaseVariable, { ...v, value: '' }))));
            expect(gen.next().done).toBeTruthy();
        });

        it('should handle null workflowId delete case', () => {
            const payload = {
                actionName: DELETE_BUTTON_ACTION,
                managedApplicationId: 'appId',
            };
            const action = { payload, type: 'any' };
            const gen: SagaIterator = navigateToWorkflowSaga(action);
            expect(gen.next().value).toStrictEqual(call(httpDELETE, `api/v1/managed-application/${payload.managedApplicationId}`));
            expect(gen.next().value).toStrictEqual(
                call(
                    toaster.info,
                    `Selected managed application has no connected ${payload.actionName} workflow,
            this action triggered deletion of Managed Application only in the Digital.a Release system`,
                ),
            );
            expect(gen.next().value).toStrictEqual(put(loadManagedApplications()));
            expect(gen.next().done).toBeTruthy();
        });

        it('should handle null workflowId update case', () => {
            const payload = {
                actionName: EDIT_BUTTON_ACTION,
                managedApplicationId: managedApplication.id,
            };
            const action = { payload, type: 'any' };
            const gen: SagaIterator = navigateToWorkflowSaga(action);
            expect(gen.next().value).toStrictEqual(call(toaster.error, `Selected managed application has no connected edit workflow`));

            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('loadEditPermissionAction', () => {
        it('should get app pipelines edit permision from local storage and update state', () => {
            const action = { payload: folder, type: 'any' };
            const gen: SagaIterator = loadEditPermissionAction(action);
            expect(gen.next().value).toStrictEqual(call(hasAnyOfPermissions, [Permissions.EDIT_APP_PIPELINES], folder.$metadata));
            const hasEditPermission = true;
            expect(gen.next(hasEditPermission).value).toStrictEqual(put(setEditPermission(hasEditPermission)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('createOrUpdateVariable', () => {
        it('should handle create variable case', () => {
            const gen: SagaIterator = createOrUpdateVariable(
                managedApplication.updateWorkflowId,
                [],
                appIdVariable.key,
                managedApplication.id,
                'xlrelease.StringVariable',
            );
            expect(gen.next([]).value).toStrictEqual(
                call(httpPOST, `api/v1/releases/Applications/${managedApplication.updateWorkflowId}/variables`, {
                    id: null,
                    key: appIdVariable.key,
                    requiresValue: false,
                    showOnReleaseStart: false,
                    type: 'xlrelease.StringVariable',
                    value: managedApplication.id,
                }),
            );
            expect(gen.next(appIdVariable).done).toBeTruthy();
        });

        it('should handle update variable case', () => {
            const gen: SagaIterator = createOrUpdateVariable(
                managedApplication.updateWorkflowId,
                [appIdVariable],
                appIdVariable.key,
                managedApplication.id,
                'xlrelease.StringVariable',
            );
            expect(gen.next().value).toStrictEqual(call(updateReleaseVariable, { ...appIdVariable, value: managedApplication.id }));
            expect(gen.next().done).toBeTruthy();
        });
    });
});
