import React, { MouseEvent, useState, useEffect } from 'react';
import ReactMarkdown from 'react-markdown';
import isString from 'lodash/isString';
import isNil from 'lodash/isNil';
import remarkGfm from 'remark-gfm';
import remarkBreaks from 'remark-breaks';
import classNames from 'classnames';
import { DotButton, DotIconButton, DotInputText, DotTypography, DotTooltip } from '@digital-ai/dot-components';
import { HelpButton } from '../../../features/main-navigation/help-button/help-button';
import { mentionsTrigger, OptionSelectorInput } from '../../../features/common/components/options-selector/option-selector-input.component';
import { ReleaseVariable, User, Variable } from '../../../types';
import getAngularService from '../../../features/common/services/angular-accessor';
import { VariablesInterpolatorFactory } from '../../../features/tasks/types/angular';
import { mentionsRegex } from '../../../js/util/filters';
import './textarea-markdown.component.less';

export const MAX_LINES = 300;

export interface TextareaMarkdownProps {
    className?: string;
    enableMentions?: boolean;
    enableVariables?: boolean;
    inputId: string;
    isReadOnly?: boolean;
    isValueRequired?: boolean;
    maxRows?: number;
    name: string;
    onDelete?: (event: MouseEvent<HTMLButtonElement>) => void;
    onTextChange?: (text: string) => void;
    placeholder?: string;
    releaseVariables?: ReleaseVariable;
    rows: number;
    text: string;
    users?: User[];
    variables?: Variable[];
}

export const TextAreaMarkdown = ({
    className,
    enableMentions = false,
    enableVariables = false,
    inputId,
    isReadOnly = false,
    isValueRequired = true,
    maxRows,
    name,
    onDelete,
    onTextChange,
    placeholder,
    rows,
    text,
    users = [],
    variables,
    releaseVariables,
}: TextareaMarkdownProps) => {
    const [editing, setEditing] = useState<boolean>(false);
    const [originalText, setOriginalText] = useState<string>(text);
    const [currentText, setCurrentText] = useState<string>(text);
    const variablesInterpolator = getAngularService('VariablesInterpolator') as never as VariablesInterpolatorFactory;
    const isSaveButtonDisabled = isValueRequired && (isNil(currentText) || !currentText.trim());

    useEffect(() => {
        setOriginalText(text);
        setCurrentText(text);
    }, [text]);

    useEffect(() => {
        if (isReadOnly) {
            revertChanges();
        }
    }, [isReadOnly]);

    const saveChanges = () => {
        const trimmedText = currentText.trim();
        onTextChange && onTextChange(trimmedText);
        setCurrentText(trimmedText);
        setOriginalText(trimmedText);
        setEditing(false);
    };

    const revertChanges = () => {
        setCurrentText(originalText);
        setEditing(false);
    };

    const handleDelete = (event: MouseEvent<HTMLButtonElement>) => {
        onDelete && onDelete(event);
        setEditing(false);
    };

    const renderTextEditor = () => (
        <div>
            {enableMentions || enableVariables ? (
                <OptionSelectorInput
                    enableMentions={enableMentions}
                    enableVariables={enableVariables}
                    id={inputId}
                    maxRows={maxRows}
                    multiline={true}
                    name={name}
                    onChange={(inputValue) => setCurrentText(inputValue)}
                    rows={rows}
                    users={users}
                    value={currentText}
                    variables={variables}
                />
            ) : (
                <DotInputText
                    autoFocus
                    id={inputId}
                    maxRows={maxRows}
                    minRows={rows}
                    multiline={true}
                    name={name}
                    onChange={(e) => setCurrentText(e.target.value)}
                    value={currentText}
                />
            )}
            <div className="actions-wrapper">
                <DotButton data-testid="cancel-button" onClick={revertChanges} type="text">
                    Cancel
                </DotButton>
                <DotButton data-testid="save-button" disabled={isSaveButtonDisabled} onClick={saveChanges} type="primary">
                    Save
                </DotButton>
                <HelpButton helpLink="how-to/using-markdown-in-xl-release.html" />
            </div>
        </div>
    );

    const wrapMentionsText = (textWithMention: string) => {
        return textWithMention.replace(mentionsRegex, (match, username) => {
            const prefix = match.substring(0, match.indexOf(mentionsTrigger));
            return `${prefix}**${mentionsTrigger}${username}**`;
        });
    };

    const findUser = (username: string) => {
        return users.find((currentUser) => currentUser.username === username);
    };

    const renderMention = (user: User) => {
        return (
            <span className="mention" title={user.username}>
                @{user.fullName || user.username}
            </span>
        );
    };

    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const renderMarkDownStrongElement = ({ children, ...props }) => {
        const startsWithMention = () => enableMentions && children && isString(children[0]) && children[0].startsWith(mentionsTrigger);
        if (startsWithMention() && children.length === 1) {
            const user = findUser(children[0].substring(mentionsTrigger.length));
            if (user) {
                return renderMention(user);
            }
        }
        if (startsWithMention() && children.length === 2 && children[1].props.children[0] && isString(children[1].props.children[0])) {
            const user = findUser(children[1].props.children[0]);
            if (user) {
                return renderMention(user);
            }
        }
        return <strong {...props}>{children}</strong>;
    };

    const replaceVariables = (value: string) => {
        if (releaseVariables && value) {
            return variablesInterpolator.interpolateInText(releaseVariables, value);
        } else {
            return value;
        }
    };

    const getRemarkPlugins = (markdownText: string) => {
        // https://github.com/micromark/micromark-extension-gfm-table/issues/6
        // there is performance issue when number of lines are bigger than X
        // testing on my laptop this was an okeish performance
        const lines = (markdownText.match(/\n/g) || '').length + 1;
        return lines > MAX_LINES ? [remarkBreaks] : [remarkGfm, remarkBreaks];
    };

    const renderMarkDownPreview = () => (
        <div
            className={classNames('markdown-viewer', { disabled: isReadOnly })}
            onDoubleClick={() => !isReadOnly && setEditing(true)}
            role="button"
            tabIndex={isReadOnly ? undefined : 0}
        >
            {!isReadOnly ? (
                <div className="markdown-viewer-actions">
                    {onDelete && originalText && <DotIconButton data-testid="delete-button" iconId="delete" onClick={handleDelete} size="small" />}
                    <DotIconButton data-testid="edit-button" iconId="edit" onClick={() => setEditing(true)} size="small" />
                </div>
            ) : null}
            {currentText ? (
                <DotTooltip disablePortal={true} title={!isReadOnly ? 'Double-click to edit' : undefined}>
                    <ReactMarkdown
                        className="markdown-wrapper"
                        components={{
                            strong: renderMarkDownStrongElement,
                            // eslint-disable-next-line react/display-name,react/prop-types
                            a: ({ href, ...props }) => {
                                if (href) {
                                    const internalRelativePath = href.startsWith('#/');
                                    const internalAbsolutePath = href.startsWith(document.location.origin);
                                    const openInNewTab = !internalAbsolutePath && !internalRelativePath;
                                    if (openInNewTab) {
                                        return <a {...props} href={href} rel="noreferrer noopener" target="_blank" />;
                                    } else {
                                        return <a {...props} href={href} />;
                                    }
                                }
                                return <a {...props} />;
                            },
                        }}
                        remarkPlugins={getRemarkPlugins(currentText)}
                    >
                        {enableMentions ? replaceVariables(wrapMentionsText(currentText)) : replaceVariables(currentText)}
                    </ReactMarkdown>
                </DotTooltip>
            ) : (
                !isReadOnly && (
                    <DotTooltip disablePortal={true} title="Double-click to edit">
                        <div className="placeholder-wrapper" onDoubleClick={() => setEditing(true)}>
                            <DotTypography className="placeholder" variant="body1">
                                {placeholder}
                            </DotTypography>
                        </div>
                    </DotTooltip>
                )
            )}
        </div>
    );

    return <div className={classNames('markdown-switcher', className)}>{editing ? renderTextEditor() : renderMarkDownPreview()}</div>;
};
