import { CardSearch } from '../ducks/analytics.reducer';
import { AnalyticsCardData } from '../types';
import { SupersetListDashboardResponse } from '../ducks/analytics.saga';

export const calculateNumberOfFiltersApplied = (cardSearch: CardSearch | undefined) => {
    let numberOfFilters = 0;
    const { categories, authorNames } = cardSearch || {};
    if (categories && categories.length > 0) numberOfFilters += categories.length;
    if (authorNames && authorNames.length > 0) numberOfFilters += authorNames.length;
    return numberOfFilters;
};

export const removeCategoryFromFilters = (categoryName: string, filteredCategories: string[] | undefined) =>
    filteredCategories?.filter((currentCategory) => currentCategory !== categoryName);

export const removeAuthorFromFilters = (authorName: string, filteredAuthorNames: string[] | undefined) =>
    filteredAuthorNames?.filter((currentAuthorName) => currentAuthorName !== authorName);

export const convertArrayToString = (input: string[] | undefined): string | undefined => {
    if (input && input.length > 0) {
        return input.join(',');
    } else {
        return undefined;
    }
};

export const isFavouriteCard = (favouriteDashboards: string[], id: number) => favouriteDashboards.includes(id.toString());

export const calculateTitleAndSubtitleForEmptyState = (isFavouriteTab: boolean, isSearchApplied: boolean): { subtitle: string; title: string } => {
    if (isSearchApplied) {
        return {
            subtitle: "We couldn't find any data that matches your filter criteria.",
            title: 'No results found',
        };
    } else {
        return isFavouriteTab
            ? {
                  subtitle: 'To easily add dashboards to your favourites, simply click the star icon located next to each of the dashboards. 🌟',
                  title: 'No favourites have been added so far',
              }
            : {
                  subtitle: 'There are no available dashboards yet.',
                  title: 'No dashboards yet',
              };
    }
};

export const filterCardsBySearch = (cards: Array<AnalyticsCardData>, cardSearch?: CardSearch): Array<AnalyticsCardData> => {
    let filteredCards = cards;
    if (cardSearch) {
        const { name, authorNames, categories } = cardSearch;

        if (name) {
            filteredCards = filteredCards.filter((card) => card.title.toLowerCase().includes(name.toLowerCase()));
        }
        if (authorNames && authorNames.length > 0) {
            filteredCards = filteredCards.filter((card) => authorNames.includes(card.subtitle));
        }
        if (categories && categories.length > 0) {
            filteredCards = filteredCards.filter((card) => card.tags.some((tag) => categories.some((category) => tag.includes(category))));
        }
    }
    return filteredCards;
};

export const mapSupersetToAnalyticsCard = (list: Array<SupersetListDashboardResponse>): Array<AnalyticsCardData> => {
    return list.map((item) => ({
        id: item.id,
        tags: ['TODO'],
        title: item.dashboard_title,
        subtitle: item.changed_by_name,
        imageSrc: 'TODO',
        contentText: 'TODO',
    }));
};

export const collectAllAuthors = (list: Array<SupersetListDashboardResponse>): Array<string> => {
    const allAuthors = new Set<string>();
    list.forEach((item) => {
        allAuthors.add(item.changed_by_name);
    });
    return Array.from(allAuthors);
};

export const getFavouriteIcon = (isFavourite: boolean) => (isFavourite ? 'star-favorites-active' : 'star-favorites-default');
export const getFavouriteText = (isFavourite: boolean) => (isFavourite ? 'Remove from favourites' : 'Add to favourites');
