import { analytics, initialState } from './analytics.reducer';
import { ANALYTICS_CATEGORIES } from '../constants';

describe('analytics reducer', () => {
    const {
        init,
        filterCards,
        setCards,
        setCategories,
        setAuthors,
        setIsDrawerOpened,
        setCardSearch,
        setIsAnalyticsConfigured,
        setIntelligenceConfiguration,
        setIsLoading,
        loadDashboard,
        setSelectedDashboard,
        clearSelectedDashboard,
    } = analytics.actions;

    it('should have the initial state', () => {
        expect(analytics.name).toBe('analytics');
        expect(analytics.reducer(undefined, { type: 'default' })).toBe(initialState);
        expect(Object.keys(analytics.actions)).toStrictEqual([
            'init',
            'filterCards',
            'setCards',
            'setCategories',
            'setAuthors',
            'setIsDrawerOpened',
            'setIsAnalyticsConfigured',
            'setIntelligenceConfiguration',
            'setCardSearch',
            'setIsLoading',
            'loadDashboard',
            'setSelectedDashboard',
            'clearSelectedDashboard',
        ]);
    });

    describe('reducers', () => {
        describe('init', () => {
            it('should NOT update state', () => {
                const state = analytics.reducer(initialState, init({ name: 'abc', authorNames: ['bob'] }));
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('filterCards', () => {
            it('should not update state', () => {
                const state = analytics.reducer(initialState, filterCards({ name: 'abc', authorNames: ['bob'] }));
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('setCards', () => {
            it('should update state', () => {
                const cards = [
                    {
                        contentText: 'Website Traffic Report for October',
                        id: 1,
                        imageSrc: 'test',
                        isFavorite: true,
                        subtitle: 'Digital.ai',
                        tags: [ANALYTICS_CATEGORIES[0], ANALYTICS_CATEGORIES[1], ANALYTICS_CATEGORIES[2]],
                        title: 'Onboarding plan and progress A',
                    },
                ];
                const state = analytics.reducer(initialState, setCards(cards));
                expect(state.cards).toStrictEqual(cards);
            });
        });
        describe('setCategories', () => {
            it('should update state', () => {
                const categories = ['cat1'];
                const state = analytics.reducer(initialState, setCategories(categories));
                expect(state.categories).toStrictEqual(categories);
            });
        });
        describe('setAuthors', () => {
            it('should update state', () => {
                const authors = ['bob', 'john'];
                const state = analytics.reducer(initialState, setAuthors(authors));
                expect(state.authors).toStrictEqual(authors);
            });
        });
        describe('setIsDrawerOpened', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setIsDrawerOpened(true));
                expect(state.isDrawerOpened).toBe(true);
            });
        });

        describe('setIsAnalyticsConfigured', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setIsAnalyticsConfigured(false));
                expect(state.isAnalyticsConfigured).toBe(false);
            });
        });

        describe('setIntelligenceConfiguration', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setIntelligenceConfiguration({} as never));
                expect(state.isAnalyticsConfigured).toBe(true);
                expect(state.intelligenceConfiguration).toStrictEqual({});
            });
        });

        describe('setCardSearch', () => {
            it('should update state', () => {
                const cardSearch = { name: 'abc', authorNames: ['bob'], categories: ['test'] };
                const state = analytics.reducer(initialState, setCardSearch(cardSearch));
                expect(state.cardSearch).toStrictEqual(cardSearch);
            });
        });

        describe('setIsLoading', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setIsLoading(true));
                expect(state.isLoading).toStrictEqual(true);
            });
        });
        describe('loadDashboard', () => {
            it('should not update state', () => {
                const state = analytics.reducer(initialState, loadDashboard(222));
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('setSelectedDashboard', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setSelectedDashboard({} as never));
                expect(state.selectedDashboard).toStrictEqual({});
            });
        });
        describe('clearSelectedDashboard', () => {
            it('should update state', () => {
                const state = analytics.reducer({ ...initialState, selectedDashboard: 'a' } as never, clearSelectedDashboard());
                expect(state.selectedDashboard).toBeUndefined();
            });
        });
    });
});
