import React, { MouseEvent, useState } from 'react';

import { DotButton, DotDialog, DotIcon, DotIconButton, DotMenu, DotTypography, MenuItemProps } from '@digital-ai/dot-components';
import { ActionToolbarPortal } from '../../../../../../../../../../core/xlr-ui/app/features/main-navigation/action-toolbar/components/action-toolbar.portal';
import { profile } from '../../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.reducer';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { UserProfileSettings } from '../../../../../../../../../../core/xlr-ui/app/types';
import { getProfile } from '../../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.selectors';
import { MenuItem } from '../../../../../../../../../../core/xlr-ui/app/react/components/menu-item/menu-item.component';
import { getFavouriteIcon, getFavouriteText } from '../../helpers';

const { toggleFavouriteDashboard } = profile.actions;

interface AnalyticsDashboardPagePortalProps {
    dashboardId: number;
}

const MENU_ITEMS_KEYS = {
    share: 'share',
    export: 'export',
};

export const AnalyticsDashboardPagePortal = ({ dashboardId }: AnalyticsDashboardPagePortalProps) => {
    const [isDialogOpen, setDialogOpen] = useState(false);
    const [isOpenDropdown, setOpenDropdown] = useState(false);
    const [modalTitle, setModalTitle] = useState('');
    const [dropdownAnchorEl, setDropdownAnchorEl] = useState<HTMLElement | null>(null);
    const dispatch = useAppDispatch();
    const profileSettings: UserProfileSettings = useAppSelector(getProfile);

    const isFavourite = profileSettings.favouriteDashboards.includes(dashboardId.toString());
    const menuOptions: Array<MenuItemProps> = [
        {
            children: <MenuItem iconId="share" label="Share dashboard" />,
            key: MENU_ITEMS_KEYS.share,
        },
        {
            children: <MenuItem iconId="download" label="Export dashboard" />,
            key: MENU_ITEMS_KEYS.export,
        },
    ];

    const handleClick = (title: string) => {
        setModalTitle(title);
        setDialogOpen(true);
    };

    const handleCloseModal = () => {
        setDialogOpen(false);
        setModalTitle('');
    };

    const switchDropdown = (event: MouseEvent<HTMLElement>) => {
        setDropdownAnchorEl(event.currentTarget);
        setOpenDropdown(!isOpenDropdown);
    };

    const handleBackClick = () => {
        window.location.href = '#/analytics';
    };

    const handleSelect = (itemKey: string) => {
        if (itemKey === MENU_ITEMS_KEYS.share) {
            handleClick('Share');
        } else if (itemKey === MENU_ITEMS_KEYS.export) {
            handleClick('Export');
        } else {
            // this should never happen, just here for now....
            handleClick('TODO');
        }
    };

    const handleFavourite = () => {
        dispatch(toggleFavouriteDashboard(dashboardId.toString()));
    };

    return (
        <ActionToolbarPortal>
            <DotButton data-testid="back-btn" onClick={handleBackClick} startIcon={<DotIcon iconId="back" />} type="text">
                Back
            </DotButton>
            <DotButton data-testid="fav-btn" onClick={handleFavourite} startIcon={<DotIcon iconId={getFavouriteIcon(isFavourite)} />} type="outlined">
                {getFavouriteText(isFavourite)}
            </DotButton>
            <DotButton data-testid="edit-btn" endIcon={<DotIcon iconId="open-new-tab" />} onClick={() => handleClick('Edit dashboard')} type="outlined">
                Edit dashboard
            </DotButton>

            <DotIconButton data-testid="options-btn" iconId="options" onClick={(e) => switchDropdown(e)} />
            <DotMenu
                anchorEl={dropdownAnchorEl || undefined}
                dense
                disablePortal
                id="analytics-portal-menu"
                menuItems={menuOptions}
                menuPlacement="bottom-end"
                onLeave={() => setOpenDropdown(false)}
                onSelect={(_e, _menu, key) => handleSelect(key)}
                open={isOpenDropdown}
            />
            <DotDialog
                cancelButtonVisible={false}
                closeIconVisible={true}
                onCancel={handleCloseModal}
                onSubmit={handleCloseModal}
                open={isDialogOpen}
                submitButtonProps={{ label: 'Close' }}
                title={modalTitle}
            >
                <DotTypography variant="body1">
                    Thanks for being curious about this feature. We are working on it right now. 🚀
                    <br />
                    Stay tuned for updates!
                </DotTypography>
            </DotDialog>
        </ActionToolbarPortal>
    );
};
