import React from 'react';
import { DotButton, DotCard, DotCardContent, DotCardFooter, DotCardHeader, DotIconButton, DotTypography } from '@digital-ai/dot-components';
import { AnalyticsCard, AnalyticsCardProps, NOT_CONFIGURED_TOOLTIP } from './analytics-card.component';
import { mockResizeObserver, mountWithTheme, ReactWrapper } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ContextMenu } from '../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/overview/components/context-menu/context-menu.component';
import { MenuItem } from '../../../../../../../../../core/xlr-ui/app/react/components/menu-item/menu-item.component';
import { CardContextMenuItemKey } from '../types';
import { ChipGroup } from '../../../../../../../../../core/xlr-ui/app/react/components/chip-group/chip-group.component';

describe('AnalyticsCard', () => {
    const id = 123;
    const contentText = 'My content text';
    const imageSrc = 'image src';
    const isFavorite = false;
    const onContextMenuItemClick = jest.fn();
    const onFavoriteClick = jest.fn();
    const onOpenDashboardClick = jest.fn();
    const subtitle = 'My subtitle';
    const tags = ['tag1', 'tag2'];
    const title = 'My card title';

    let wrapper: ReactWrapper;

    const defaultProps: AnalyticsCardProps = {
        contentText,
        id,
        imageSrc,
        isAnalyticsConfigured: true,
        isFavorite,
        onContextMenuItemClick,
        onFavoriteClick,
        onOpenDashboardClick,
        subtitle,
        tags,
        title,
    };

    const mount = (props: AnalyticsCardProps = defaultProps) => {
        wrapper = mountWithTheme(<AnalyticsCard {...props} />);
    };

    beforeAll(() => {
        mockResizeObserver();
    });

    afterAll(() => {
        jest.restoreAllMocks();
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    const getDotCard = () => wrapper.find(DotCard);
    const getDotCardHeader = () => getDotCard().find(DotCardHeader);
    const getDotCardContent = () => getDotCard().find(DotCardContent);
    const getFavoriteButton = () => getDotCardHeader().findWhere((node) => node.is(DotIconButton) && node.props()['data-testid'] === 'favorite-button');
    const getContextMenu = () => getDotCardHeader().find(ContextMenu);
    const getImage = () => getDotCard().find('img');
    const getContentTypography = () => getDotCardContent().findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'content-typography');
    const getChipGroup = () => getDotCardContent().find(ChipGroup);
    const getDotCardFooter = () => getDotCard().find(DotCardFooter);
    const getOpenDashboardButton = () => getDotCardFooter().findWhere((node) => node.is(DotButton) && node.props().className === 'open-dashboard-btn');

    describe('default render', () => {
        it('should render with the default props', () => {
            mount();
            expect(getDotCard()).toExist();

            const dotCardHeader = getDotCardHeader();
            expect(dotCardHeader).toExist();
            expect(dotCardHeader.props().className).toBe('card-header');
            expect(dotCardHeader.props().subheader).toBe(`by ${subtitle}`);
            expect(dotCardHeader.props().title).toBe(title);
            expect(dotCardHeader.props().titleSize).toBe('medium');

            const dotCardContent = getDotCardContent();
            expect(dotCardContent).toExist();
            expect(dotCardContent.props().className).toBe('card-content');

            const favoriteButton = getFavoriteButton();
            expect(favoriteButton).toExist();
            expect(favoriteButton.props().iconId).toBe('star-favorites-default');
            expect(favoriteButton.props().tooltip).toBe('Add to favourites');

            const contextMenu = getContextMenu();
            expect(contextMenu.props().menuItems).toStrictEqual([
                {
                    children: <MenuItem endIconId="open-new-tab" iconId="edit" label="Edit" />,
                    key: CardContextMenuItemKey.Edit,
                },
                {
                    children: <MenuItem iconId="visibility-off" label="Hide" />,
                    key: CardContextMenuItemKey.Hide,
                },
                {
                    children: <MenuItem iconId="duplicate" label="Duplicate" />,
                    key: CardContextMenuItemKey.Duplicate,
                },
                {
                    children: <MenuItem iconId="share" label="Share" />,
                    key: CardContextMenuItemKey.Share,
                },
                {
                    children: <MenuItem iconId="download" label="Export" />,
                    key: CardContextMenuItemKey.Export,
                },
            ]);

            const image = getImage();
            expect(image).toExist();
            expect(image.prop('alt')).toBe('Anaylics card image');
            expect(image.prop('src')).toBe(imageSrc);

            const contentTypography = getContentTypography();
            expect(contentTypography).toExist();
            expect(contentTypography.props().variant).toBe('body1');
            expect(contentTypography.props().children).toBe(contentText);

            const chipGroup = getChipGroup();
            expect(chipGroup).toExist();
            expect(chipGroup.props().labels).toBe(tags);

            const cardFooter = getDotCardFooter();
            expect(cardFooter).toExist();

            const openDashboardButton = getOpenDashboardButton();
            expect(openDashboardButton).toExist();
            expect(openDashboardButton.props().type).toBe('outlined');
        });

        it('should change favorite icon and tooltip when favorite is set to true', () => {
            mount({ ...defaultProps, isFavorite: true });
            expect(getFavoriteButton().props().iconId).toBe('star-favorites-active');
            expect(getFavoriteButton().props().tooltip).toBe('Remove from favourites');
        });

        it('should trigger correct handler when context item is selected', () => {
            mount();
            const contextMenu = getContextMenu();
            contextMenu.invoke('onSelect')('edit');
            expect(onContextMenuItemClick).toHaveBeenCalledTimes(1);
            expect(onContextMenuItemClick).toHaveBeenCalledWith('edit');
        });

        it('should trigger correct handler when Open dashboard button is clicked', () => {
            mount();
            const openDashboardBtn = getOpenDashboardButton();
            openDashboardBtn.invoke('onClick')();
            expect(onOpenDashboardClick).toHaveBeenCalledTimes(1);
        });
    });

    describe('when analytics is not configured', () => {
        beforeEach(() => {
            mount({ ...defaultProps, isAnalyticsConfigured: false });
        });

        it('should disable favourite button', () => {
            const favButton = getFavoriteButton();
            expect(favButton.props().tooltip).toBe(NOT_CONFIGURED_TOOLTIP);
            expect(favButton.props().disabled).toBe(true);
        });

        it('should disable context menu', () => {
            const contextMenu = getContextMenu();
            expect(contextMenu.props().tooltip).toBe(NOT_CONFIGURED_TOOLTIP);
            expect(contextMenu.props().disabled).toBe(true);
        });

        it('should add data-testid to open dashboard button', () => {
            const openDashboardButton = getOpenDashboardButton();
            expect(openDashboardButton.props()['data-testid']).toBe('open-analytics-dashboard-btn');
        });
    });
});
