import {
    generateEnvironmentTitle,
    getCommitHashFromCommitUrl,
    getDeepLink,
    getTrimmedCommitHashFromCommitUrl,
    getVersionLabel,
    getVersionLink,
    getVersionTooltip,
    mapOrderByValue,
} from './utils';
import {
    APPLICATION_COLUMN_ID,
    APPLICATION_ORDER_BY,
    DESTINATION_COLUMN_ID,
    DESTINATION_ORDER_BY,
    STATUS_COLUMN_ID,
    STATUS_ORDER_BY,
    TIME_COLUMN_ID,
    TIME_ORDER_BY,
    VERSION_COLUMN_ID,
    VERSION_ORDER_BY,
} from '../constants';
import { deploymentStateMock, deploymentStateMockDeploy } from '../__mocks__/external-deployments.mocks';

describe('utils', () => {
    describe('getDeepLink', () => {
        it('should combine endpointUrl with applicationPath', () => {
            expect(getDeepLink('endp', 'oint')).toStrictEqual('endpoint');
            expect(getDeepLink('endp', undefined)).toStrictEqual('endp');
        });
    });

    describe('mapOrderByValue', () => {
        it('should map columnName to correct orderBy value', () => {
            expect(mapOrderByValue(APPLICATION_COLUMN_ID)).toStrictEqual(APPLICATION_ORDER_BY);
            expect(mapOrderByValue(DESTINATION_COLUMN_ID)).toStrictEqual(DESTINATION_ORDER_BY);
            expect(mapOrderByValue(STATUS_COLUMN_ID)).toStrictEqual(STATUS_ORDER_BY);
            expect(mapOrderByValue(VERSION_COLUMN_ID)).toStrictEqual(VERSION_ORDER_BY);
            expect(mapOrderByValue(TIME_COLUMN_ID)).toStrictEqual(TIME_ORDER_BY);
        });
        it('should map unknown columnName to default orderBy of STATE', () => {
            expect(mapOrderByValue('other')).toStrictEqual(STATUS_ORDER_BY);
        });
    });

    describe('getTrimmedCommitHashFromCommitUrl', () => {
        it('should get trimmed commit hash from commit url', () => {
            expect(
                getTrimmedCommitHashFromCommitUrl('https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a'),
            ).toStrictEqual('d654f0c');
            expect(getTrimmedCommitHashFromCommitUrl('https://github.com/mkeretic1/argocd-sandbox/commit/d654f0')).toStrictEqual('d654f0');
        });
    });

    describe('getCommitHashFromCommitUrl', () => {
        it('should get commit hash from commit url', () => {
            expect(getCommitHashFromCommitUrl('https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a')).toStrictEqual(
                'd654f0c68f9213b19818ba23d91350886362431a',
            );
        });
    });

    describe('getVersionLabel', () => {
        it('should trim ArgoCD commit version label', () => {
            expect(getVersionLabel(true, 'https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a')).toStrictEqual(
                '#d654f0c',
            );
        });
        it('should trim ArgoCD helm version label', () => {
            expect(getVersionLabel(true, 'https://bitnami-labs.github.io/sealed-secrets/2.13.1')).toStrictEqual('2.13.1');
        });
        it('should not trim Deploy version label', () => {
            expect(getVersionLabel(false, '1.0')).toStrictEqual('1.0');
        });
    });

    describe('getVersionTooltip', () => {
        it('should return commit hash from commit url for ArgoCD git app', () => {
            expect(getVersionTooltip(true, 'https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a')).toStrictEqual(
                'd654f0c68f9213b19818ba23d91350886362431a',
            );
        });
        it('should return version label for ArgoCD helm app', () => {
            expect(getVersionTooltip(true, 'https://bitnami-labs.github.io/sealed-secrets/2.13.1')).toStrictEqual(
                'https://bitnami-labs.github.io/sealed-secrets',
            );
        });
        it('should return version label for Deploy', () => {
            expect(getVersionTooltip(false, '1.0')).toStrictEqual('1.0');
        });
    });

    describe('getVersionLink', () => {
        it('should return commit url for ArgoCD git app', () => {
            expect(
                getVersionLink(true, 'https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a', 'http://deploy', '/app1'),
            ).toStrictEqual('https://github.com/mkeretic1/argocd-sandbox/commit/d654f0c68f9213b19818ba23d91350886362431a');
        });
        it('should return commit url for ArgoCD helm app', () => {
            expect(getVersionLink(true, 'https://bitnami-labs.github.io/sealed-secrets/2.13.1', 'http://deploy', '/app1')).toStrictEqual(
                'https://bitnami-labs.github.io/sealed-secrets',
            );
        });
        it('should return deep link for Deploy', () => {
            expect(getVersionLink(false, '1.0', 'http://deploy', '/app1')).toStrictEqual('http://deploy/app1');
        });
    });

    describe('generateEnvironmentTitle', () => {
        it('should generate environment title for state and type, default is empty', () => {
            expect(generateEnvironmentTitle(deploymentStateMockDeploy)).toStrictEqual('Environment');
            expect(generateEnvironmentTitle(deploymentStateMock)).toStrictEqual('Environment/namespace');
        });
    });
});
