import React, { memo, useEffect, useState } from 'react';
import isNil from 'lodash/isNil';
import { DotAutoComplete, DotButton, DotCard, DotCardContent, DotCardHeader, DotThemeProvider, DotCheckbox, DotIcon } from '@digital-ai/dot-components';
import { CreateServerForm } from '../create-server/application-create-server-component';
import { DiscoverModal } from '../discover-modal/discover-modal-component';
import { Server, WebhookSource, WebhookSourceAutoConfigDetails } from '../../../external-deployment.types';
import { Folder } from '../../../../../../../../../../../core/xlr-ui/app/types';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import './application-discovery-basic.component.less';
import { FilterApplications } from '../filter-applications/filter-applications-component';
import { folderExternalDeployments } from '../../../ducks/external-deployments.reducer';
import {
    getConfigDetailsSelector,
    getIsLoadingSelector,
    getValidServerCards,
    getPreSelectedServer,
    getServersSelector,
    getWebhookSourcesSelector,
} from '../../../ducks/external-deployments.selectors';
import { STATUS_HTTP_CONNECTION } from '../../../constants';

const { loadServers, loadWebhookSources, getAvailablePlugins } = folderExternalDeployments.actions;

export interface ApplicationDiscoveryBasicComponentProp {
    closeWizard: () => void;
    folder: Folder;
}
export interface PluginInfo {
    iconLocation: string;
    isAutoconfigurable: boolean;
    subheader: string;
    title: string;
    type: string;
}

const { setConfigDetails, setLiveUpdate, setPreSelectedServer } = folderExternalDeployments.actions;

const ApplicationDiscoveryBasicComponent = (props: ApplicationDiscoveryBasicComponentProp) => {
    const { folder, closeWizard } = props;

    const configDetails: WebhookSourceAutoConfigDetails | undefined = useAppSelector(getConfigDetailsSelector);
    const isLoading: boolean = useAppSelector(getIsLoadingSelector);
    const preSelected = useAppSelector(getPreSelectedServer);
    const servers: Array<Server> = useAppSelector(getServersSelector);
    const webhookSources: Array<WebhookSource> = useAppSelector(getWebhookSourcesSelector);
    const validServerCards: PluginInfo[] = useAppSelector(getValidServerCards);
    const [autoCreatedWebhook, setAutoCreatedWebhook] = useState<WebhookSource | undefined>(undefined);

    const [selectedServers, setSelectedServers] = useState<{
        [key in string]?: Server;
    }>({});

    const [typeOfCreatedServer, setTypeOfCreatedServer] = useState<string | undefined>(undefined);
    const [discoverType, setDiscoverType] = useState<string | undefined>(undefined);
    const [selectedServer, setSelectedServer] = useState<Server | undefined>(undefined);
    const [createManagedApplications, setCreateManagedApplications] = useState<boolean>(false);
    const [isModalOpen, setIsModalOpen] = useState<boolean>(false);
    const [isFilterAppsOpen, setIsFilterAppsOpen] = useState<boolean>(false);

    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(setPreSelectedServer(undefined));
        dispatch(loadServers(folder.id));
        dispatch(loadWebhookSources(folder.id));
        dispatch(getAvailablePlugins(STATUS_HTTP_CONNECTION));
    }, []);

    useEffect(() => {
        if (!isNil(preSelected)) {
            updateSelectedServer(preSelected, preSelected.type);
        } else {
            setSelectedServers({});
        }
    }, [preSelected]);

    useEffect(() => {
        if (!configDetails) return;

        !configDetails.success && configDetails.webhookSourceId ? onModalSubmit() : setIsModalOpen(true);
    }, [configDetails]);

    const updateSelectedServer = (server: Server, serverType: string) => {
        setSelectedServers((prevState) => ({ ...prevState, [serverType]: server }));
        setDiscoverType(serverType);
    };

    const getSelectedServer = (serverType: string) => {
        return selectedServers[serverType];
    };

    const checkIfSelected = (serverType: string) => {
        return !isNil(selectedServers[serverType]);
    };

    const handleDiscoverApplication = (serverType: string) => {
        setSelectedServer(getSelectedServer(serverType));
        const server = getSelectedServer(serverType);

        if (server?.id) {
            dispatch(setLiveUpdate({ folderId: folder.id, serverId: server.id }));
        }
    };

    const onModalSubmit = () => {
        setAutoCreatedWebhook(webhookSources.find((webhookSource: WebhookSource) => webhookSource?.id === configDetails?.webhookSourceId));
        setIsModalOpen(false);
        setIsFilterAppsOpen(true);
        dispatch(setConfigDetails(undefined));
    };

    const closeModal = () => {
        dispatch(setConfigDetails(undefined));
        setIsModalOpen(false);
    };

    const filterBack = () => {
        setIsFilterAppsOpen(false);
    };

    const generateServerCard = (pluginInfo: PluginInfo) => {
        const serverOptions: Array<Server> = servers.filter((server: Server) => server.type === pluginInfo.type);
        const onCreateServer = () => setTypeOfCreatedServer(pluginInfo.type);
        return (
            <DotCard>
                <DotCardHeader
                    avatar={pluginInfo.iconLocation.length > 0 ? <img alt={pluginInfo.type} src={pluginInfo.iconLocation} /> : <DotIcon iconId={'plugins'} />}
                    className="card-header-inline"
                    subheader={pluginInfo.subheader}
                    subheaderSize="small"
                    title={pluginInfo.title}
                    titleSize="small"
                />
                <DotCardContent>
                    {serverOptions.length === 0 && (
                        <DotButton className="discovery-button" data-testid="create-server-btn" onClick={onCreateServer}>
                            Create connection
                        </DotButton>
                    )}
                    {serverOptions.length > 0 && (
                        <DotAutoComplete
                            actionItem={{
                                iconId: 'add',
                                onClick: onCreateServer,
                                text: 'Create connection',
                            }}
                            className="server-source"
                            freesolo={false}
                            inputId="server-source-input"
                            label="Choose connection"
                            multiple={false}
                            onChange={(e, v) => updateSelectedServer(v as Server, pluginInfo.type)}
                            options={serverOptions}
                            value={{ title: getSelectedServer(pluginInfo.type)?.title || '' }}
                        />
                    )}
                    {serverOptions.length > 0 && (
                        <DotButton
                            className="discovery-button"
                            data-testid="discover-server-btn"
                            disabled={!checkIfSelected(pluginInfo.type)}
                            onClick={() => handleDiscoverApplication(pluginInfo.type)}
                        >
                            Discover applications
                        </DotButton>
                    )}
                </DotCardContent>
            </DotCard>
        );
    };

    const generateServerCards = () => {
        return (
            <div className="discovery">
                <h2>Discover applications</h2>
                <div className="server-cards">{validServerCards.map((plugin) => generateServerCard(plugin))}</div>
                {configDetails && (
                    <DiscoverModal
                        configureDetails={configDetails}
                        isModalOpen={isModalOpen && !isLoading}
                        onModalClose={closeModal}
                        onModalSubmit={onModalSubmit}
                        type={discoverType || ''}
                    />
                )}
                <DotCheckbox
                    checked={createManagedApplications}
                    className="discover-managed-applications-checkbox"
                    data-testid="discover-managed-applications-checkbox"
                    label="Also create managed applications from discovered applications"
                    onChange={() => setCreateManagedApplications(!createManagedApplications)}
                />
                <DotButton data-testid="discovery-cancel-btn" onClick={closeWizard} type="text">
                    Cancel
                </DotButton>
            </div>
        );
    };

    const generateBasicView = () => {
        return (
            <>
                {!typeOfCreatedServer && generateServerCards()}
                {typeOfCreatedServer && (
                    <CreateServerForm closeForm={() => setTypeOfCreatedServer(undefined)} folder={folder} serverType={typeOfCreatedServer} />
                )}
            </>
        );
    };

    return (
        <DotThemeProvider>
            {!isFilterAppsOpen && generateBasicView()}
            {isFilterAppsOpen && (
                <FilterApplications
                    createManagedApplications={createManagedApplications}
                    onCancel={closeWizard}
                    onFilterBack={filterBack}
                    onSave={closeWizard}
                    plugins={validServerCards}
                    server={selectedServer}
                    webhookSource={autoCreatedWebhook as WebhookSource}
                />
            )}
        </DotThemeProvider>
    );
};

export const BasicDiscovery = memo(ApplicationDiscoveryBasicComponent);
