import React, { memo, useState, useEffect, KeyboardEvent } from 'react';
import { DotAvatar, DotButton, DotCard, DotCardContent, DotIcon, DotIconButton, DotMenu, DotPill, DotTooltip, MenuItemProps } from '@digital-ai/dot-components';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';
import { ManagedApplicationType } from '../../managed-application.types';
import './managed-application.component.less';
import { PluginInfo } from '../../../external-deployments/components/wizard/basic-wizard/application-discovery-basic.component';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getValidServerCards } from '../../ducks/managed-application.selectors';
import { STATUS_HTTP_CONNECTION } from '../../../external-deployments/constants';
import { folderExternalDeployments } from '../../../external-deployments/ducks/external-deployments.reducer';
import { Folder } from '../../../../../../../../../../core/xlr-ui/app/types';
import { navigation } from '../../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { applicationManagement } from '../../ducks/managed-application.reducer';
import { extractContentInLastBrackets } from '../../helper/utils';

const { getAvailablePlugins, setLiveDeploymentsPreconditionFilter } = folderExternalDeployments.actions;
const { navigateToWorkflow } = applicationManagement.actions;
const { navigateTo } = navigation.actions;

export interface ManagedApplicationProps {
    folder: Folder;
    managedApplication: ManagedApplicationType;
}

export const EDIT_BUTTON_ACTION = 'edit';
export const DELETE_BUTTON_ACTION = 'delete';

const ManagedApplicationComponent = ({ folder, managedApplication }: ManagedApplicationProps) => {
    const isManagedByRelease = managedApplication.managedBy === 'Release';
    const { name: targetConnection, type: serverType } = managedApplication.connectionServer;
    const validServerCards: PluginInfo[] = useAppSelector(getValidServerCards);

    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(getAvailablePlugins(STATUS_HTTP_CONNECTION));
    }, []);

    const serverIcon = validServerCards.find((validServerCard) => validServerCard.type === serverType)?.iconLocation || '';
    const [open, setOpen] = useState(false);
    const [anchorEl, setAnchorEl] = useState<undefined | HTMLElement>(undefined);
    const handleToggle = (event: React.MouseEvent<HTMLButtonElement, MouseEvent>) => {
        setAnchorEl(event.currentTarget);
        setOpen(!open);
    };
    const menuItems: MenuItemProps[] = [
        { key: EDIT_BUTTON_ACTION, children: 'Edit' },
        { key: DELETE_BUTTON_ACTION, children: 'Delete' },
    ];

    const onItemSelect =
        (managedApplicationId: string, deleteWorkflowId?: string, updateWorkflowId?: string) =>
        (_event: React.MouseEvent<Element, MouseEvent> | KeyboardEvent<Element>, _menuId: string, itemKey: string) => {
            setOpen(false);
            if (itemKey === DELETE_BUTTON_ACTION) {
                dispatch(
                    navigateToWorkflow({
                        actionName: itemKey,
                        managedApplicationId,
                        workflowId: deleteWorkflowId,
                    }),
                );
            } else if (itemKey === EDIT_BUTTON_ACTION) {
                dispatch(
                    navigateToWorkflow({
                        actionName: itemKey,
                        managedApplicationId,
                        workflowId: updateWorkflowId,
                    }),
                );
            }
        };

    const viewDeployments = (app: string) => {
        dispatch(setLiveDeploymentsPreconditionFilter({ folderId: folder.id, applicationName: app }));
        dispatch(navigateTo({ pathSuffix: `folders/${folder.id}/application-pipelines/external-deployments` }));
    };

    return (
        <DotCard className="created-application-card" key={managedApplication.applicationName}>
            <div data-testid={`app${managedApplication.applicationName}`}>
                <DotCardContent>
                    <div className="application-name-column" title={managedApplication.applicationName}>
                        {managedApplication.applicationName}
                    </div>

                    <div className="managed-by-column">
                        <DotTooltip title={managedApplication.managedBy}>
                            <DotPill
                                backgroundcolor={isManagedByRelease ? '#E8EDF3 !important' : '#F3F5F6 !important'}
                                bordercolor={isManagedByRelease ? '#3D6C9E' : undefined}
                                icon={<DotIcon iconId={isManagedByRelease ? 'release' : 'lock'} />}
                                label={upperFirst(managedApplication.managedBy)}
                                labelcolor="#3B485C !important"
                                size="small"
                                status="success"
                                variant="outlined"
                            />
                        </DotTooltip>
                    </div>

                    <div className="date-created-column">{moment(managedApplication.dateCreated).format('MMM DD, YYYY')}</div>

                    <div className="target-deployment-column">
                        <DotAvatar alt="ArgoCD Logo" imageSrc={serverIcon} size="small" />
                        <div className="target-deployment-column-text">
                            <DotTooltip title={extractContentInLastBrackets(managedApplication.applicationReference)}>
                                <div className="target-deployment-column-text">{extractContentInLastBrackets(managedApplication.applicationReference)}</div>
                            </DotTooltip>
                        </div>
                    </div>

                    <div className="environment-tag-column">
                        <DotPill
                            backgroundcolor="#E9F0E0 !important"
                            bordercolor="#498500"
                            icon={<DotIcon iconId="environment" />}
                            label={managedApplication.environmentTag}
                            labelcolor="#3B485C !important"
                            size="small"
                            status="success"
                            variant="outlined"
                        />
                    </div>

                    <div className="target-connection-column">{targetConnection}</div>

                    <div className="link-column">
                        <DotButton onClick={() => viewDeployments(managedApplication.applicationName)} tooltip="View deployments" type="outlined">
                            View deployments
                        </DotButton>
                    </div>

                    <div className="actions-column">
                        <div className="actions">
                            <DotIconButton iconId="options" onClick={handleToggle} />
                        </div>
                        <DotMenu
                            anchorEl={anchorEl}
                            disablePortal={true}
                            id="actions-menu"
                            menuItems={menuItems}
                            onLeave={() => setOpen(false)}
                            onSelect={onItemSelect(managedApplication.id, managedApplication.deleteWorkflowId, managedApplication.updateWorkflowId)}
                            open={open}
                        />
                    </div>
                </DotCardContent>
            </div>
        </DotCard>
    );
};

export const ManagedApplication = memo(ManagedApplicationComponent);
