import React, { memo, useEffect, useState } from 'react';
import isNil from 'lodash/isNil';
import { ENDPOINT_TYPE, SupportedServerType } from '../../../constants';
import { DotAutoComplete, DotButton, DotCard, DotCardContent, DotCardHeader, DotThemeProvider } from '@digital-ai/dot-components';
import { getDiscoveryCardSubheader, getDiscoveryCardTitle, getServerIcon } from '../../../helper/utils';
import { CreateServerForm } from '../create-server/application-create-server-component';
import { DiscoverModal } from '../discover-modal/discover-modal-component';
import { Server, WebhookSource, WebhookSourceAutoConfigDetails } from '../../../external-deployment.types';
import { Folder } from '../../../../../../../../../../../core/xlr-ui/app/types';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import './application-discovery-basic.component.less';
import { FilterApplications } from '../filter-applications/filter-applications-component';
import { folderExternalDeployments } from '../../../ducks/external-deployments.reducer';
import {
    getConfigDetailsSelector,
    getIsLoadingSelector,
    getPreSelectedServer,
    getServersSelector,
    getWebhookSourcesSelector,
} from '../../../ducks/external-deployments.selectors';

const { loadServers, loadWebhookSources } = folderExternalDeployments.actions;

export interface ApplicationDiscoveryBasicComponentProp {
    closeWizard: () => void;
    folder: Folder;
}

const { setConfigDetails, setLiveUpdate, setPreSelectedServer } = folderExternalDeployments.actions;

const ApplicationDiscoveryBasicComponent = (props: ApplicationDiscoveryBasicComponentProp) => {
    const { folder, closeWizard } = props;

    const configDetails: WebhookSourceAutoConfigDetails | undefined = useAppSelector(getConfigDetailsSelector);
    const isLoading: boolean = useAppSelector(getIsLoadingSelector);
    const preSelected = useAppSelector(getPreSelectedServer);
    const servers: Array<Server> = useAppSelector(getServersSelector);
    const webhookSources: Array<WebhookSource> = useAppSelector(getWebhookSourcesSelector);

    const [autoCreatedWebhook, setAutoCreatedWebhook] = useState<WebhookSource | undefined>(undefined);
    const [selectedDeployServer, setSelectedDeployServer] = useState<Server | undefined>(undefined);
    const [selectedArgoServer, setSelectedArgoServer] = useState<Server | undefined>(undefined);
    const [typeOfCreatedServer, setTypeOfCreatedServer] = useState<SupportedServerType | undefined>(undefined);
    const [discoverType, setDiscoverType] = useState<string | undefined>(undefined);
    const [selectedServer, setSelectedServer] = useState<Server | undefined>(undefined);

    const [isModalOpen, setIsModalOpen] = useState<boolean>(false);
    const [isFilterAppsOpen, setIsFilterAppsOpen] = useState<boolean>(false);

    const dispatch = useAppDispatch();

    const argoServers: Array<Server> = servers.filter((server: Server) => server.type === ENDPOINT_TYPE.argoCd);
    const deployServers: Array<Server> = servers.filter((server: Server) => server.type === ENDPOINT_TYPE.deploy);

    useEffect(() => {
        if (!isNil(preSelected)) {
            updateSelectedServer(preSelected, preSelected.type);
        } else {
            setSelectedDeployServer(undefined);
            setSelectedArgoServer(undefined);
        }
    }, [preSelected]);

    useEffect(() => {
        dispatch(loadServers(folder.id));
        dispatch(loadWebhookSources(folder.id));
        dispatch(setPreSelectedServer(undefined));
    }, []);

    useEffect(() => {
        if (!configDetails) return;

        !configDetails.success && configDetails.webhookSourceId ? onModalSubmit() : setIsModalOpen(true);
    }, [configDetails]);

    const updateSelectedServer = (server: Server, serverType: SupportedServerType) => {
        if (serverType === ENDPOINT_TYPE.deploy) setSelectedDeployServer(server);
        else setSelectedArgoServer(server);
        setDiscoverType(serverType);
    };

    const getSelectedServer = (serverType: SupportedServerType) => {
        if (serverType === ENDPOINT_TYPE.deploy) return selectedDeployServer;
        else return selectedArgoServer;
    };

    const checkIfSelected = (serverType: SupportedServerType) => {
        if (serverType === ENDPOINT_TYPE.deploy) return !isNil(selectedDeployServer);
        else return !isNil(selectedArgoServer);
    };

    const handleDiscoverApplication = (serverType: SupportedServerType) => {
        setSelectedServer(getSelectedServer(serverType));
        const server = getSelectedServer(serverType);

        if (server?.id) {
            dispatch(setLiveUpdate({ folderId: folder.id, serverId: server.id }));
        }
    };

    const onModalSubmit = () => {
        setAutoCreatedWebhook(webhookSources.find((webhookSource: WebhookSource) => webhookSource?.id === configDetails?.webhookSourceId));
        setIsModalOpen(false);
        setIsFilterAppsOpen(true);
        dispatch(setConfigDetails(undefined));
    };

    const closeModal = () => {
        dispatch(setConfigDetails(undefined));
        setIsModalOpen(false);
    };

    const filterBack = () => {
        setIsFilterAppsOpen(false);
    };

    const generateServerCard = (serverType: SupportedServerType, serverOptions: Array<Server>) => {
        const onCreateServer = () => setTypeOfCreatedServer(serverType);
        return (
            <DotCard>
                <DotCardHeader
                    avatar={<img alt={serverType} src={getServerIcon(serverType)} />}
                    className="card-header-inline"
                    subheader={getDiscoveryCardSubheader(serverType)}
                    subheaderSize="small"
                    title={getDiscoveryCardTitle(serverType)}
                    titleSize="small"
                />
                <DotCardContent>
                    {serverOptions.length > 0 && (
                        <DotAutoComplete
                            actionItem={{
                                iconId: 'add',
                                onClick: onCreateServer,
                                text: 'Create connection',
                            }}
                            className="server-source"
                            freesolo={false}
                            inputId="server-source-input"
                            label="Choose connection"
                            multiple={false}
                            onChange={(e, v) => updateSelectedServer(v as Server, serverType)}
                            options={serverOptions}
                            value={{ title: getSelectedServer(serverType)?.title || '' }}
                        />
                    )}
                    {serverOptions.length === 0 && (
                        <DotButton className="discovery-button" data-testid="create-server-btn" onClick={onCreateServer}>
                            Create connection
                        </DotButton>
                    )}
                    {serverOptions.length > 0 && (
                        <DotButton
                            className="discovery-button"
                            data-testid="discover-server-btn"
                            disabled={!checkIfSelected(serverType)}
                            onClick={() => handleDiscoverApplication(serverType)}
                        >
                            Discover applications
                        </DotButton>
                    )}
                </DotCardContent>
            </DotCard>
        );
    };

    const generateServerCards = () => {
        return (
            <div className="discovery">
                <h2>Discover applications</h2>
                <div className="server-cards">
                    {generateServerCard(ENDPOINT_TYPE.argoCd, argoServers)}
                    {generateServerCard(ENDPOINT_TYPE.deploy, deployServers)}
                </div>
                {configDetails && (
                    <DiscoverModal
                        configureDetails={configDetails}
                        isModalOpen={isModalOpen && !isLoading}
                        onModalClose={closeModal}
                        onModalSubmit={onModalSubmit}
                        type={discoverType || ''}
                    />
                )}
                <DotButton data-testid="discovery-cancel-btn" onClick={closeWizard} type="text">
                    Cancel
                </DotButton>
            </div>
        );
    };

    const generateBasicView = () => {
        return (
            <>
                {!typeOfCreatedServer && generateServerCards()}
                {typeOfCreatedServer && (
                    <CreateServerForm closeForm={() => setTypeOfCreatedServer(undefined)} folder={folder} serverType={typeOfCreatedServer} />
                )}
            </>
        );
    };

    return (
        <DotThemeProvider>
            {!isFilterAppsOpen && generateBasicView()}
            {isFilterAppsOpen && (
                <FilterApplications
                    onCancel={closeWizard}
                    onFilterBack={filterBack}
                    onSave={closeWizard}
                    server={selectedServer}
                    webhookSource={autoCreatedWebhook as WebhookSource}
                />
            )}
        </DotThemeProvider>
    );
};

export const BasicDiscovery = memo(ApplicationDiscoveryBasicComponent);
