package com.xebialabs.xlrelease.assertions;

import java.util.Map;
import org.assertj.core.api.Assertions;
import org.assertj.core.data.MapEntry;

import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.ReleaseTrigger;
import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.domain.variables.Variable;
import java.util.Arrays;

import static com.xebialabs.xlrelease.assertions.XLReleaseAssertions.assertThat;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.ABORTED;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.COMPLETED;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.FAILED;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.FAILING;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.IN_PROGRESS;
import static com.xebialabs.xlrelease.variable.VariableHelper.withVariableSyntax;

import static org.assertj.core.data.MapEntry.entry;

public class ReleaseAssertion extends PlanItemAssertion<ReleaseAssertion, Release> {

    protected ReleaseAssertion(Release actual) {
        super(actual, ReleaseAssertion.class);
    }

    public ReleaseAssertion hasStatus(ReleaseStatus status) {
        isNotNull();
        Assertions.assertThat(actual.getStatus()).isEqualTo(status);
        return this;
    }

    public ReleaseAssertion isCompleted() {
        isNotNull().hasStatus(COMPLETED).hasEndDate().hasFlagStatus(FlagStatus.OK).hasFlagComment("");
        return this;
    }

    public ReleaseAssertion isInProgress() {
        isNotNull().hasStatus(IN_PROGRESS).hasNoEndDate();
        return this;
    }

    public ReleaseAssertion isFailing() {
        isNotNull().hasStatus(FAILING);
        return this;
    }

    public ReleaseAssertion isFailed() {
        isNotNull().hasStatus(FAILED);
        return this;
    }

    public ReleaseAssertion isAborted() {
        isNotNull().hasStatus(ABORTED);
        return this;
    }

    public ReleaseAssertion hasOwner(String owner) {
        isNotNull();
        Assertions.assertThat(actual.getOwner()).isEqualTo(owner);
        return this;
    }

    public ReleaseAssertion hasDescription(String description) {
        isNotNull();
        Assertions.assertThat(actual.getDescription()).isEqualTo(description);
        return this;
    }

    public ReleaseAssertion hasTitle(String title) {
        isNotNull();
        Assertions.assertThat(actual.getTitle()).isEqualTo(title);
        return this;
    }

    public ReleaseAssertion containsVariables(Variable... variables) {
        isNotNull();

        MapEntry[] entries = new MapEntry[variables.length];
        for (int i = 0; i < variables.length; i++) {
            Variable variable = variables[i];
            entries[i] = entry(withVariableSyntax(variable.getKey()), variable.getValue());
        }
        Assertions.assertThat(actual.getVariableValues()).contains(entries);
        return this;
    }

    public ReleaseAssertion hasTags(String... tags) {
        isNotNull();
        Assertions.assertThat(actual.getTags()).containsExactly(tags);
        return this;
    }

    public ReleaseAssertion hasPrivateCalendar() {
        isNotNull();
        Assertions.assertThat(actual.getCalendarLinkToken()).isNotEmpty();
        Assertions.assertThat(actual.isCalendarPublished()).isFalse();
        return this;
    }

    public PlanItemAssertion hasAbortOnFailure(boolean abortOnFailure) {
        isNotNull();
        Assertions.assertThat(actual.isAbortOnFailure()).isEqualTo(abortOnFailure);
        return this;
    }

    public PlanItemAssertion hasAllowConcurrentReleasesFromTrigger(boolean allowConcurrentReleaseFromTrigger) {
        isNotNull();
        Assertions.assertThat(actual.isAllowConcurrentReleasesFromTrigger()).isEqualTo(allowConcurrentReleaseFromTrigger);
        return this;
    }

    public ReleaseAssertion hasOriginTemplateId(String originTemplateId) {
        isNotNull();
        Assertions.assertThat(actual.getOriginTemplateId()).isEqualTo(originTemplateId);
        return this;
    }

    public ReleaseAssertion hasRunningTriggeredReleasesCount(int runningTriggeredReleasesCount) {
        isNotNull();
        Assertions.assertThat(actual.getRunningTriggeredReleasesCount()).isEqualTo(runningTriggeredReleasesCount);
        return this;
    }

    public ReleaseAssertion hasOnlyDisabledTriggers() {
        isNotNull();
        for (ReleaseTrigger releaseTrigger : actual.getReleaseTriggers()) {
            assertThat(releaseTrigger).hasNoExecutionId();
            assertThat(releaseTrigger).isDisabled();
        }
        return this;
    }

    public ReleaseAssertion hasRealFlagStatus(FlagStatus releaseFlagStatus) {
        isNotNull();
        Assertions.assertThat(actual.getRealFlagStatus()).isEqualTo(releaseFlagStatus);
        return this;
    }

    public ReleaseAssertion hasScriptUsername(String scriptUserName) {
        isNotNull();
        Assertions.assertThat(actual.getScriptUsername()).isEqualTo(scriptUserName);
        return this;
    }

    public ReleaseAssertion hasScriptUserPassword(String scriptUserPassword) {
        isNotNull();
        Assertions.assertThat(actual.getScriptUserPassword()).isEqualTo(scriptUserPassword);
        return this;
    }

    public ReleaseAssertion containsOnlyMemberViewers(String... members){
        isNotNull();
        Assertions.assertThat(actual.getMemberViewers()).containsOnly(members);
        return this;
    }

    public ReleaseAssertion containsOnlyRoleViewers(String... roles){
        isNotNull();
        Assertions.assertThat(actual.getRoleViewers()).containsOnly(roles);
        return this;
    }

    public ReleaseAssertion hasPasswordVariables(Map<String, String> variables) {
        isNotNull();
        Assertions.assertThat(actual.getPasswordVariableValues()).isEqualTo(variables);
        return this;
    }

    public ReleaseAssertion hasVariable(String variableKey) {
        isNotNull();
        Assertions.assertThat(actual.getVariablesByKeys().containsKey(variableKey));
        return this;
    }

    public ReleaseAssertion hasVariables(String ... variableKeys) {
        isNotNull();
        Assertions.assertThat(actual.getVariablesByKeys().keySet()).containsAll(Arrays.asList(variableKeys));
        return this;
    }

}
