package com.xebialabs.xlrelease

import scala.reflect.ClassTag

/**
 * This is a base trait to be inherited by ScalaTest classes which want to run an integration test
 * with XL Release. Inheritors of this class can [[org.springframework.beans.factory.annotation.Autowire]]
 * XL Release services into fields and use them to setup a test environment and assert conditions.
 * <p>
 * A typical test would look like this:
 *
 * {{{
 * class MyIntegrationTest extends XLReleaseIntegrationScalaTest {
 *
 *   &#064;Autowired
 *   private var repositoryService: RepositoryService = _
 *
 *   describe("MyIntegrationTest") {
 *
 *     it("should create release in repository") {
 *       val release = ReleaseBuilder.newRelease().withId(TestIds.RELEASE1).build();
 *       repositoryService.create(release)
 *       repositoryService.read[Release](TestIds.RELEASE1) should not be null
 *     }
 *   }
 * }
 * }}}
 *
 * </p>
 * <p>
 * <strong>Note:</strong> A test instance of XL Release is setup in a temporary folder and
 * has some services mocked up for speed and easier testing. So you cannot test initializers or
 * upgraders, for example.
 * </p>
 *
 */
trait XLReleaseIntegrationScalaTest extends XLReleaseIntegrationTest with XLReleaseScalaTest {

  override protected def beforeAll(): Unit = {
    super.beforeAll()
    before()
  }

  override protected def beforeEach(): Unit = {
    super.beforeEach()
    before()
    directory.before()
  }

  override protected def afterEach(): Unit = {
    super.afterEach()
    tearDown()
    directory.after()
  }

  def springBean[T: ClassTag]: T = applicationContext.getBean(implicitly[ClassTag[T]].runtimeClass).asInstanceOf[T]
}
