#
# Copyright 2017 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import sys
import string
import time
import com.xhaus.jyson.JysonCodec as json
from vsts.HttpRequest import HttpRequest
from vsts.util import error
from vsts.TfsServer import TfsServer

class ReleaseClient(TfsServer):
    """Exceptions are documented in the same way as classes.

    The __init__ method may be documented in either the class level
    docstring, or as a docstring on the __init__ method itself.

    Either form is acceptable, but the two should not be mixed. Choose one
    convention to document the __init__ method and be consistent with it.
    """

    __json = 'application/json'
    __apiVersion = 'api-version=3.0-preview.2'

    def __init__(self, tfs_server, username = None, password = None, domain = None):
        if tfs_server is None:
            error('No server provided.')

        url = tfs_server['url']
        vsts = 'visualstudio.com'

        if vsts in url:
            index = url.find(vsts)
            tfs_server['url'] = url[:index] + 'vsrm.' + url[index:]

        if tfs_server["azureCloud"]:
            index = url.find("://") + 3
            tfs_server['url'] = url[:index] + 'vsrm.' + url[index:]

        self.request = HttpRequest(tfs_server, username, password, domain)

    def createRelease(self, teamProjectName, releaseName, releaseDescription, buildNumber):
        releaseDefinitionId = self.getReleaseDefinitionId(teamProjectName, releaseName)
        artifacts = self.getDefaultArtifacts(teamProjectName, releaseDefinitionId, buildNumber, releaseDescription)

        url = '%s/_apis/release/releases?%s' % (teamProjectName, self.__apiVersion)

        response = self.request.post(url, json.dumps(artifacts), contentType=self.__json)

        if response.status != 200:
            error("Error creating release.", response)

        release = json.loads(response.response)

        return release['id']

    def getReleaseDefinitionId(self, teamProjectName, releaseName):
        if not teamProjectName:
            raise Exception("Team project name is a mandatory field.")

        if not releaseName:
            raise Exception("Release Name is a mandatory field.")

        response = self.request.get('%s/_apis/release/definitions?searchText=%s' % (teamProjectName, releaseName))

        if response.status == 401 or response.status == 403:
            error("Authentication failed", response)
        
        if response.status != 200:
            error("Incorrect Project or Release name", response)

        release = json.loads(response.response)
        
        if not release['value'] or not release['value'][0]:
            error("Incorrect Release name '%s'" % releaseName, response)
            
        releaseId = None
        for value in release['value']:
            if value['name'] == releaseName:
                releaseId = value['id']
                
        if not releaseId:
            error("Incorrect Release name '%s'" % releaseName, response)
            
        return releaseId

    def getDefaultArtifacts(self, teamProjectName, releaseDefinitionId, buildNumber, releaseDescription = ""):
        response = self.request.get('%s/_apis/Release/artifacts/versions?releaseDefinitionId=%s' % (teamProjectName, releaseDefinitionId))

        artifacts = json.loads(response.response)

        return ReleaseStartMetadata(artifacts, releaseDefinitionId, buildNumber, releaseDescription)

    def getReleaseStatus(self, teamProjectName, releaseId, environmentName):
        release = self.getReleaseById(teamProjectName, releaseId)
        status = None

        for environment in release['environments']:
            if environment['name'].upper() == environmentName.upper():
                status = environment['status']

        if status is None:
            error("No environment with name '%s' found" % environmentName)

        return status

    def getReleaseById(self, teamProjectName, releaseId):
        if not releaseId:
            error('ReleaseId is a mandatory field. No Release Id supplied.')

        response = self.request.get('%s/_apis/release/releases/%s' % (teamProjectName, releaseId))

        if response.status != 200:
            error("Release with id '%s' was not found." % releaseId, response)

        return json.loads(response.response)

    def getReleaseEnvironmentId(self, teamProjectName, releaseId, environmentName):
        release = self.getReleaseById(teamProjectName, releaseId)
        id = None

        for environment in release['environments']:
            if environment['name'].upper() == environmentName.upper():
                id = environment['id']

        if id is None:
            error("No environment with name '%s' found" % environmentName)

        return id

    def startDeployment(self, teamProjectName, releaseId, environmentName, comment):
        envId = self.getReleaseEnvironmentId(teamProjectName, releaseId, environmentName)

        url = '%s/_apis/release/releases/%s/environments/%s?%s' % (teamProjectName, releaseId, envId, self.__apiVersion)
        body = '{"status": "inprogress","comment":"%s"}' % comment

        response = self.request.patch(url, body, contentType=self.__json)

        if response.status != 200:
            error("Failed to start the deployment for the environment '%s'." % environmentName, response)

    def getApprovals(self, teamProjectName, releaseId):
        url = '%s/_apis/release/approvals?releaseIdsFilter=%s&%s' % (teamProjectName, releaseId, self.__apiVersion)

        response = self.request.get(url)

        if response.status != 200:
            error("Failed to retrieve the approvals for the release '%s'." % releaseId)

        return json.loads(response.response)

    def approveDeployment(self, teamProjectName, releaseId, environmentName, comment):
        if not releaseId:
            error('ReleaseId is a mandatory field. No Release Id supplied.')

        attempts = 0

        while attempts < 3:
            approvals = self.getApprovals(teamProjectName, releaseId)

            if (approvals['count'] == 0):
                if (attempts == 2):
                    error("No approvals available for the release '%s'." % releaseId)
                else:
                    time.sleep(5)
                    attempts += 1
            else:
                break

        approvalId = None

        for approval in approvals['value']:
            if approval['releaseEnvironment']['name'].upper() == environmentName.upper():
                approvalId = approval['id']

        if approvalId is None:
            error("No approvals found for the environment '%s'" % environmentName)

        url = '%s/_apis/release/approvals/%s?%s' % (teamProjectName, approvalId, self.__apiVersion)
        body = '{"status": "approved","comment":"%s"}' % comment

        response = self.request.patch(url, body, contentType=self.__json)

        if response.status != 200:
            error("Failed to approve the deployment for the environment '%s'." % environmentName, response)

class ReleaseStartMetadata(object):
    def __init__(self, artifacts, releaseDefinitionId, buildNumber, releaseDescription = ""):

        self.definitionId = releaseDefinitionId
        self.description = releaseDescription
        self.buildNumber = buildNumber
        self.artifacts = []

        for item in artifacts['artifactVersions']:
            artifact = ArtifactMetadata(item, buildNumber)
            self.artifacts.append(artifact)

    def __json__(self):
        artifacts = []
        for a in self.artifacts:
            artifacts.append(a.__json__())

        test = ",".join(artifacts).join(("[","]"))

        return u"""{"definitionId":%s, "description":"%s", "artifacts":%s}""" % (self.definitionId, self.description, test)

class ArtifactMetadata(object):
    def __init__(self, artifacts, buildVersionInput):
        self.alias = artifacts['alias']
        version = None
        definition_name = None

        default_version = artifacts['defaultVersion']

        if artifacts['versions']:
            latest_version = artifacts['versions'][0]
            if default_version is None:
                default_version = latest_version;
            definition_name = latest_version["definitionName"]
        else:
            error("Build runs not available for '%s'." % self.alias)

        if buildVersionInput is None:
            version = default_version

        elif buildVersionInput and definition_name in buildVersionInput:
            build_version_str = buildVersionInput[definition_name]

            for a in artifacts['versions']:
                if a["name"] == build_version_str:
                    version = a
                    break
            if version is None:
                error("Incorrect build number for '%s'." % self.alias)

        elif buildVersionInput and definition_name not in buildVersionInput:
            version = default_version

        else:
            version = default_version

        self.instance_reference = BuildVersion(version)

    def __json__(self):
        return u"""{"alias":"%s", "instanceReference":%s }""" % (self.alias, self.instance_reference.__json__())

class BuildVersion(object):
    def __init__(self, version):
        try:
            self.id = version['id']
            self.name = version['name']
            self.source_branch = version['sourceBranch']
            self.is_repo_artifact = True if 'commitMessage' in version.keys() else False
            self.commit_message = version['commitMessage'] if 'commitMessage' in version.keys() else ""
            self.source_repository_id = version['sourceRepositoryId'] if 'sourceRepositoryId' in version.keys() else ""
            self.source_repository_type = version['sourceRepositoryType'] if 'sourceRepositoryType' in version.keys() else ""
            self.source_version = version['sourceVersion'] if 'sourceVersion' in version.keys() else ""
        except:
            error("Build version missing values")


    def __json__(self):
        return u"""{"id":"%s", "name":"%s", "sourceBranch":"%s", "sourceVersion":"%s", "sourceRepositoryId":"%s", "sourceRepositoryType":"%s"}""" % (self.id, self.name, self.source_branch, self.source_version, self.source_repository_id, self.source_repository_type)