#
# Copyright (c) 2019. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import json
import org.apache.http.conn.HttpHostConnectException

from datetime import datetime
from xlrelease.HttpRequest import HttpRequest
from distutils.version import LooseVersion

PAGE_SIZE = 500


class SonarApiService(object):
    metrics = {}

    @staticmethod
    def get_sonar_metrics(sonar_server):
        if len(SonarApiService.metrics) < 1:
            http_request = SonarApiService.get_http_request(sonar_server)
            sonar_response = http_request.get('api/metrics/search?ps=500')
            json_data = json.loads(sonar_response.getResponse())

            for metric in json_data['metrics']:
                SonarApiService.metrics[metric['key']] = metric

        return SonarApiService.metrics

    @staticmethod
    def get_project_metrics(sonar_server, project_id, branch, pullRequest):
        sonar_server_api_url = "/api/qualitygates/project_status?projectKey={}".format(project_id)
        if branch:
            sonar_server_api_url = sonar_server_api_url+'&branch='+branch
        elif pullRequest:
            sonar_server_api_url = sonar_server_api_url+'&pullRequest='+pullRequest
        http_request = SonarApiService.get_http_request(sonar_server)
        sonar_response = http_request.get(sonar_server_api_url)
        if not sonar_response.isSuccessful():
            raise Exception(
                "Failed to get Sonar Measures. Server returned response [%s] with status code [%s]" % (
                    sonar_response.response, sonar_response.status
                ))
        return json.loads(sonar_response.getResponse())

    @staticmethod
    def get_analysis_date(sonar_server, project_id, branch, pullRequest):
        sonar_server_api_url = "/api/components/show?component={}".format(project_id)
        if branch:
            sonar_server_api_url = sonar_server_api_url+'&branch='+branch
        elif pullRequest:
            sonar_server_api_url = sonar_server_api_url+'&pullRequest='+pullRequest
        http_request = SonarApiService.get_http_request(sonar_server)
        sonar_response = http_request.get(sonar_server_api_url)
        if not sonar_response.isSuccessful():
            raise Exception(
                "Failed to get analysis date. Server returned response [%s] with status code [%s]" % (
                    sonar_response.response, sonar_response.status
                ))
        return json.loads(sonar_response.getResponse())['component']['analysisDate']

    @staticmethod
    def format_sonar_api_date(date):
        return datetime.fromtimestamp(date / 1000.0).strftime('%Y-%m-%d')

    @staticmethod
    def get_data_by_path(sonar_server, api_path):
        http_request = SonarApiService.get_http_request(sonar_server)
        sonar_response = http_request.get(api_path)
        if sonar_response.isSuccessful():
            json_data = json.loads(sonar_response.getResponse())
            return json_data
        else:
            error = json.loads(sonar_response.getResponse())
            print error
            return {'error': error['errors']}

    @staticmethod
    def get_search_history(sonar_server, date_from_str, date_to_str, resource, metrics, page_number):
        search_history_api_url = "/api/measures/search_history?component={}&ps={}&p={}&from={}&to={}&metrics={}".format(
            resource, PAGE_SIZE, page_number, date_from_str, date_to_str, ','.join(metrics.keys()))
        return SonarApiService.get_data_by_path(sonar_server, search_history_api_url)

    @staticmethod
    def get_project_analyses(sonar_server, date_from_str, date_to_str, resource, page_number):
        project_analyses_api_url = "/api/project_analyses/search?project={}&p={}&ps={}&from={}&to={}&category=VERSION".format(
            resource, page_number, PAGE_SIZE, date_from_str, date_to_str)
        return SonarApiService.get_data_by_path(sonar_server, project_analyses_api_url)

    @staticmethod
    def paginate_sonar_request(request_func, filter_func):
        try:
            data = {'data': []}
            page = 1
            while page:
                sonar_response = request_func(page)
                data.get('data').extend(filter_func(sonar_response))
                if sonar_response.get("paging").get("total") <= page * PAGE_SIZE:
                    page = False
                else:
                    page += 1
        except org.apache.http.conn.HttpHostConnectException as e:
            print "Connection Failed : {0} ".format(str(e))
            data = {'error': [{'msg': e.getMessage()}]}
        return data

    @staticmethod
    def get_http_request(sonar_server):
        if sonar_server['authenticationMethod']:
            sonar_server['authenticationMethod'] = str(sonar_server['authenticationMethod'])
        if "userToken" in sonar_server and sonar_server["userToken"]:
            sonar_server['authenticationMethod'] = 'Bearer'
            return HttpRequest(sonar_server, username='', password=sonar_server["userToken"])
        else:
            return HttpRequest(sonar_server)

    @staticmethod
    def get_sonar_component_metrics(sonar_server, resource, branch, pullRequest):
        http_request = SonarApiService.get_http_request(sonar_server)
        sonar_version_api = "/api/server/version"
        version_response = str(http_request.get(sonar_version_api).getResponse())
        if LooseVersion(version_response) >= LooseVersion("8.0.0"):
            sonar_server_api_url = '/api/measures/component?component=%s&metricKeys=complexity,line_coverage,duplicated_lines_density,violations' % resource
        else:
            sonar_server_api_url = '/api/measures/component?componentKey=%s&metricKeys=complexity,line_coverage,duplicated_lines_density,violations' % resource
        if branch:
            sonar_server_api_url = sonar_server_api_url+'&branch='+branch
        elif pullRequest:
            sonar_server_api_url = sonar_server_api_url+'&pullRequest='+pullRequest

        print (sonar_server_api_url)
        return http_request.get(sonar_server_api_url)

    @staticmethod
    def get_sonar_blocking_violation_metrics(sonar_server, resource, branch, pullRequest):
        metrics = ['blocker_violations','critical_violations','major_violations']
        http_request = SonarApiService.get_http_request(sonar_server)
        sonar_server_api_url = '/api/measures/component?component=%s&metricKeys=%s' % (resource, ','.join(metrics))
        if branch:
            sonar_server_api_url = sonar_server_api_url+'&branch='+branch
        elif pullRequest:
            sonar_server_api_url = sonar_server_api_url+'&pullRequest='+pullRequest

        print (sonar_server_api_url)
        return http_request.get(sonar_server_api_url)

    @staticmethod
    def search_project_analysis(sonar_server, resource, branch, pullRequest):
        http_request = SonarApiService.get_http_request(sonar_server)
        sonar_server_api_url = '/api/project_analyses/search?project=%s&p=1&ps=1' % resource
        if branch:
            sonar_server_api_url = sonar_server_api_url+'&branch='+branch
        elif pullRequest:
            sonar_server_api_url = sonar_server_api_url+'&pullRequest='+pullRequest
        print (sonar_server_api_url)
        return http_request.get(sonar_server_api_url)

    @staticmethod
    def updateEvent(sonar_server, event_key,version):
        http_request = SonarApiService.get_http_request(sonar_server)
        sonar_server_api_url = '/api/project_analyses/update_event?event=%s&name=%s' % (event_key, version)
        return http_request.post(sonar_server_api_url,body="")

    @staticmethod
    def createEvent(sonar_server, analysis_key,version):
        http_request = SonarApiService.get_http_request(sonar_server)
        sonar_server_api_url = '/api/project_analyses/create_event?analysis=%s&category=VERSION&name=%s' % (analysis_key, version)
        return http_request.post(sonar_server_api_url,body="")