#
# Copyright 2024 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import com.xhaus.jyson.JysonCodec as json
from xlrelease.HttpRequest import HttpRequest

from slack.util import error
from slack.util import validate_mandatory_field


class SlackServer(object):
    __json = 'application/json'

    def __init__(self, slack_server):
        validate_mandatory_field(slack_server, "Slack Server not configured")
        self.request = HttpRequest(slack_server)

    def create_channel(self, channel):
        validate_mandatory_field(channel, "Channel not configured")

        request_body = {
            'name': channel
        }

        response = self.request.post('/conversations.create', json.dumps(request_body), contentType=self.__json)
        if response.status != 200:
            error("Unexpected error happened while creating a channel", response)

        response_body = json.loads(response.response)
        if response_body['ok']:
            return response_body['channel']['id']
        else:
            error("Could not create channel", response)

    def archive_channel(self, channel_id):
        validate_mandatory_field(channel_id, "Channel ID not configured")

        request_body = {
            'channel': channel_id
        }

        response = self.request.post('/conversations.archive', json.dumps(request_body), contentType=self.__json)
        if response.status != 200:
            error("Unexpected error happened while archiving a channel", response)

        response_body = json.loads(response.response)
        if not response_body['ok']:
            error("Could not archive channel", response)

    def send_message(self, channel, message, icon=None, username=None):
        validate_mandatory_field(channel, "Channel not configured")
        validate_mandatory_field(message, "Message not configured")

        request_body = {
            'channel': channel,
            'text': message
        }

        if username:
            request_body['username'] = username

        if icon:
            request_body['icon_emoji'] = icon

        response = self.request.post('/chat.postMessage', json.dumps(request_body), contentType=self.__json)
        if response.status != 200:
            error("Unexpected error happened while sending a message", response)

        response_body = json.loads(response.response)
        if not response_body['ok']:
            error("Could not post message", response)

    def add_users_to_channel(self, channel_id, user_ids):
        validate_mandatory_field(channel_id, "Channel ID not configured")
        validate_mandatory_field(user_ids, "User IDs not configured")

        request_body = {
            'channel': channel_id,
            'users': ",".join(user_ids)
        }

        response = self.request.post('/conversations.invite', json.dumps(request_body), contentType=self.__json)
        if response.status != 200:
            error("Unexpected error happened while inviting users to channel", response)

        response_body = json.loads(response.response)
        if not response_body['ok']:
            error("Could not invite users to channel", response)
