package com.xebialabs.xlrelease.plugin.overthere;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Timer;
import java.util.TimerTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionLogService;
import com.xebialabs.xlrelease.storage.service.LogSizeLimitExceededException;

class TaskExecutionLogHelper extends BufferedOutputStream {
    private static final Logger logger = LoggerFactory.getLogger(TaskExecutionLogHelper.class);
    private final Timer timer;
    private volatile boolean running = true;

    public TaskExecutionLogHelper(Task task, TaskExecutionLogOutputStream outputStream, Duration pollingInterval) {
        super(outputStream, TaskExecutionLogService.DEFAULT_CHUNK_BUFFER_SIZE());

        this.timer = new Timer("task-execution-log-" + task.getExecutionId() + "-" + task.getFailuresCount(), true);
        this.timer.schedule(new TimerTask() {
                                @Override
                                public void run() {
                                    flushBuffer();
                                }
                            },
                0,
                pollingInterval.toMillis());
    }

    public void log(String line) {
        if (!running) {
            logger.debug("Log helper no longer running, skipping the line");
            return;
        }
        try {
            this.write(line.getBytes(StandardCharsets.UTF_8));
        } catch (IOException e) {
            logger.error("Failed to log a line", e);
        } catch (LogSizeLimitExceededException ex) {
            logger.debug("Log size limit exceeded", ex);
            stop();
        }
    }

    public void flushBuffer() {
        try {
            flush();
        } catch (IOException e) {
            logger.error("Failed to flush the buffer", e);
        }
    }

    public void stop() {
        this.timer.cancel();
        this.timer.purge();
        this.running = false;
    }

    @Override
    public void close() throws IOException {
        stop();
        super.close();
    }

}
