package com.xebialabs.xlrelease.plugin.overthere;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionLogService;
import com.xebialabs.xlrelease.storage.domain.LogEntry;
import com.xebialabs.xlrelease.storage.service.LogSizeLimitExceededException;

import scala.Option;

class TaskExecutionLogHelper implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(TaskExecutionLogHelper.class);
    private final Task task;
    private final int bufferSize;
    private final Timer timer;
    private final TaskExecutionLogService taskExecutionLogService;
    private final AtomicInteger chunk = new AtomicInteger(0);
    private final ByteArrayOutputStream byteArrayOutputStream;
    private final BufferedOutputStream outputStream;
    private volatile boolean running = true;

    private int currentBufferSize = 0;

    public TaskExecutionLogHelper(Task task, TaskExecutionLogService taskExecutionLogService) {
        this.task = task;
        // buffer size will translate directly into a file chunk on a file system - so let's make it slightly bigger than task.getMaxCommentSize()
        bufferSize = TaskExecutionLogService.DEFAULT_CHUNK_BUFFER_SIZE();
        byteArrayOutputStream = new ByteArrayOutputStream(bufferSize * 2);
        outputStream = new BufferedOutputStream(byteArrayOutputStream, bufferSize);
        this.taskExecutionLogService = taskExecutionLogService;
        // in theory, we could use thread pool instead of a timer but there is little to no benefit
        this.timer = new Timer("task-execution-log-" + task.getExecutionId() + "-" + task.getFailuresCount(), true);
        this.timer.schedule(new TimerTask() {
                                @Override
                                public void run() {
                                    flushBuffer();
                                }
                            },
                0,
                5000);
    }

    public void log(String line) {
        if (!running) {
            logger.debug("Log helper no longer running, skipping the line");
            return;
        }
        try {
            this.append(line.getBytes(StandardCharsets.UTF_8));
        } catch (IOException e) {
            logger.error("Failed to log a line", e);
        }
    }

    public void flushBuffer() {
        try {
            outputStream.flush();

            byte[] currentBuffer = byteArrayOutputStream.toByteArray();
            byteArrayOutputStream.reset();
            currentBufferSize = 0;
            if (currentBuffer.length > 0) {
                LogEntry entry = new LogEntry(
                        task.getId(),
                        task.getExecutionId(),
                        task.getFailuresCount(), // job id == failures count
                        chunk.incrementAndGet(),
                        DateTimeFormatter.ISO_INSTANT.format(Instant.now()),
                        currentBuffer,
                        Option.empty()
                );

                taskExecutionLogService.log(entry);
            }
        } catch (IOException e) {
            logger.error("Failed to flush the buffer", e);
        } catch (LogSizeLimitExceededException ex) {
            logger.debug("Log size limit exceeded", ex);
            stop();
        }
    }

    public void stop() {
        this.timer.cancel();
        this.running = false;
    }

    private void append(byte[] payload) throws IOException {
        outputStream.write(payload);
        currentBufferSize += payload.length;

        if (currentBufferSize > bufferSize) {
            flushBuffer();
        }
    }

    @Override
    public void close() throws IOException {
        flushBuffer();
        stop();
        outputStream.close();
    }
}
