package com.xebialabs.xlrelease.plugin.overthere;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.overthere.*;
import com.xebialabs.xlplatform.spring.SpringContextHolder;
import com.xebialabs.xlrelease.domain.PythonScript;
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionLogService;

import static com.xebialabs.overthere.ConnectionOptions.JUMPSTATION;
import static com.xebialabs.overthere.ConnectionOptions.OPERATING_SYSTEM;
import static com.xebialabs.overthere.ConnectionOptions.TEMPORARY_DIRECTORY_PATH;
import static com.xebialabs.overthere.OperatingSystemFamily.UNIX;
import static com.xebialabs.overthere.cifs.CifsConnectionBuilder.WINRM_TIMEMOUT;
import static com.xebialabs.overthere.ssh.SshConnectionBuilder.CONNECTION_TYPE;
import static com.xebialabs.overthere.ssh.SshConnectionBuilder.SUDO_USERNAME;
import static com.xebialabs.overthere.ssh.SshConnectionType.SUDO;
import static com.xebialabs.overthere.util.OverthereFileTranscoder.transcode;

public class RemoteScript {
    private static final String SCRIPT_NAME = "uploaded-script";

    private final ConnectionOptions options = new ConnectionOptions();
    private final String protocol;
    private final Writer stdoutWriter;
    private final String remotePath;
    private final String script;
    private final String extension;
    private final PythonScript pythonScript;
    private String stdOut;
    private String stdErr;

    public RemoteScript(PythonScript pythonScript, Writer stdoutWriter) {
        this.pythonScript = pythonScript;
        this.protocol = pythonScript.getProperty("protocol");
        this.stdoutWriter = stdoutWriter;
        copyPropertiesToConnectionOptions(options, pythonScript);
        OperatingSystemFamily osFamily = options.get(OPERATING_SYSTEM, UNIX);
        String scriptContent = pythonScript.getProperty("script");
        this.script = scriptContent != null ? scriptContent.replaceAll("\\R", osFamily.getLineSeparator()) : null;
        this.extension = osFamily.getScriptExtension();
        this.remotePath = pythonScript.getProperty("remotePath");
    }

    protected void copyPropertiesToConnectionOptions(ConnectionOptions options, ConfigurationItem ci) {
        // support legacy properties
        if (ci.hasProperty("sudo") && (Boolean) (ci.getProperty("sudo"))) {
            ci.setProperty(CONNECTION_TYPE, SUDO);
            ci.setProperty(SUDO_USERNAME, "root");
        }

        // copy all CI properties to connection properties
        for (PropertyDescriptor pd : ci.getType().getDescriptor().getPropertyDescriptors()) {
            if (!pd.getCategory().equals("output")) {
                Object value = pd.get(ci);
                setConnectionOption(options, pd.getName(), value);
            }
        }
    }

    private void setConnectionOption(ConnectionOptions options, String key, Object value) {
        if (key.equals("script") || key.equals("remotePath") || key.equals("scriptLocation")) {
            return;
        }

        if (value == null || value.toString().isEmpty()) {
            return;
        }

        // support legacy properties
        if (key.equals("temporaryDirectoryPath")) {
            key = TEMPORARY_DIRECTORY_PATH;
        } else if (key.equals("timeout")) {
            key = WINRM_TIMEMOUT;
        }

        if (value instanceof Integer && (Integer) value == 0) {
            logger.debug("Activating workaround for DEPLOYITPB-4775: Integer with value of 0 not passed to Overthere.");
            return;
        }

        if (key.equals(JUMPSTATION)) {
            ConfigurationItem item = (ConfigurationItem) value;

            ConnectionOptions jumpstationOptions = new ConnectionOptions();
            copyPropertiesToConnectionOptions(jumpstationOptions, item);
            options.set(key, jumpstationOptions);
        } else {
            options.set(key, value);
        }
    }

    public int execute() {
        var task = pythonScript.getCustomScriptTask();
        TaskExecutionLogService taskExecutionLogService = SpringContextHolder.getApplicationContext().getBean(TaskExecutionLogService.class);
        var taskExecutionLogHelper = new TaskExecutionLogHelper(task, taskExecutionLogService);
        var passwordMasker = new PasswordMasker(task);
        var maxSize = task.getMaxCommentSize();
        try (OverthereConnection connection = Overthere.getConnection(protocol, options);
             var stdoutHandler = createOutputHandler(taskExecutionLogHelper, passwordMasker, maxSize);
             var stderrHandler = createOutputHandler(taskExecutionLogHelper, passwordMasker, maxSize)
        ) {
            return doExecute(connection, stdoutHandler, stderrHandler);
        } catch (IOException e) {
            logger.error("Unable to close output handler", e);
            return 1;
        }
    }

    private OutputHandler createOutputHandler(TaskExecutionLogHelper taskExecutionLogHelper, PasswordMasker passwordMasker, int maxSize) {
        return new PasswordMaskingOutputHandler( // 1. mask passwords
                new TaskExecutionLogOutputHandler( // 2. send to task execution log
                        new NonClosingWriterOutputHandler( // 3. send to the script engine stdout
                                new RingWriterOutputHandler(maxSize), // 4. buffer last maxCommentSize
                                stdoutWriter
                        ),
                        taskExecutionLogHelper),
                passwordMasker);
    }

    private int doExecute(OverthereConnection connection, OutputHandler stdoutHandler, OutputHandler stderrHandler) throws IOException {
        try {
            if (remotePath != null && !remotePath.isEmpty()) {
                connection.setWorkingDirectory(connection.getFile(remotePath));
            }

            // Upload and execute the script
            OverthereFile targetFile = connection.getTempFile(SCRIPT_NAME, extension);
            transcode(script, "UTF-8", targetFile);
            targetFile.setExecutable(true);

            CmdLine scriptCommand = CmdLine.build(targetFile.getPath());

            return connection.execute(stdoutHandler, stderrHandler, scriptCommand);
        } catch (Exception e) {
            var stackTrace = captureStackTrace(e);
            stderrHandler.handleLine(stackTrace);
            return 1;
        } finally {
            stdOut = getOutput(stdoutHandler);
            stdErr = getOutput(stderrHandler);
        }
    }

    private static String captureStackTrace(final Exception e) throws IOException {
        try (StringWriter stacktrace = new StringWriter();
             PrintWriter writer = new PrintWriter(stacktrace, true)
        ) {
            e.printStackTrace(writer);
            return stacktrace.toString();
        }
    }

    public String getStdout() {
        return stdOut;
    }

    public String getStderr() {
        return stdErr;
    }

    private String getOutput(OutputHandler outputHandler) throws IOException {
        return outputHandler.getStringContent();
    }

    public ConnectionOptions getOptions() {
        return options;
    }

    public String getScript() {
        return script;
    }

    private static final Logger logger = LoggerFactory.getLogger(RemoteScript.class);

}
