package com.xebialabs.xlrelease.plugin.overthere.util;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.xlrelease.domain.CustomScriptTask;
import com.xebialabs.xlrelease.domain.PythonScript;
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionLogService;
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration;
import com.xebialabs.xlrelease.storage.domain.LogEntry;

public class TaskExecutionLogHelper {
    private static final Logger logger = LoggerFactory.getLogger(TaskExecutionLogHelper.class);
    private static final int BUFFER_SIZE = 8192;

    private final PythonScript pythonScript;
    private final TaskExecutionLogService taskExecutionLogService;
    private final AtomicInteger chunk = new AtomicInteger(0);
    private final ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream(BUFFER_SIZE * 2);
    private final BufferedOutputStream outputStream = new BufferedOutputStream(byteArrayOutputStream, BUFFER_SIZE);
    private final Timer timer = new Timer();

    private int currentBufferSize = 0;

    public TaskExecutionLogHelper(PythonScript pythonScript, TaskExecutionLogService taskExecutionLogService) {
        this.pythonScript = pythonScript;
        this.taskExecutionLogService = taskExecutionLogService;
        this.timer.schedule(new TimerTask() {
                                @Override
                                public void run() {
                                    flushBuffer();
                                }
                            },
                0,
                5000);
    }

    public void log(String line) {
        // overthere removes the line endings
        String lineWithEnding = line.concat(System.getProperty("line.separator"));
        try {
            this.append(lineWithEnding.getBytes(StandardCharsets.UTF_8));
        } catch (IOException e) {
            logger.error("Failed to log a line", e);
        }
    }


    public void flushBuffer() {
        try {
            outputStream.flush();

            byte[] currentBuffer = byteArrayOutputStream.toByteArray();
            byteArrayOutputStream.reset();
            currentBufferSize = 0;
            if (currentBuffer.length > 0) {
                CustomScriptTask customScriptTask = pythonScript.getCustomScriptTask();
                taskExecutionLogService.log(
                        new LogEntry(
                                customScriptTask.getId(),
                                customScriptTask.getExecutionId(),
                                customScriptTask.getFailuresCount(),
                                chunk.incrementAndGet(),
                                DateTimeFormatter.ISO_INSTANT.format(Instant.now()),
                                currentBuffer,
                                StorageConfiguration.URI_SCHEME_LOCAL_STORAGE()
                        )
                );
            }
        } catch (IOException e) {
            logger.error("Failed to flush the buffer", e);
        }
    }

    public void stopTimer() {
        this.timer.cancel();
    }

    private void append(byte[] payload) throws IOException {
        outputStream.write(payload);
        currentBufferSize += payload.length;

        if (currentBufferSize > BUFFER_SIZE) {
            flushBuffer();
        }
    }

}
