#
# Copyright (c) XebiaLabs B.V., All rights reserved.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import sys
import os
import tempfile
import requests
from com.xebialabs.deployit.plugin.opsgenie import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption
from opsgenie.config import Configuration, ProxyConfiguration
from opsgenie.swagger_client import configuration as SwaggerConf
from urlparse import urlparse

class APIClient:

    def __init__(self, opsgenie_config):
        self.api_key = opsgenie_config['apiKey']
        self.apiEndpoint = opsgenie_config['apiEndpoint']
        self.proxy_host = opsgenie_config['proxyHost']
        self.proxy_port = opsgenie_config['proxyPort']
        self.proxy_username = opsgenie_config['proxyUsername']
        self.proxy_password = opsgenie_config['proxyPassword']
        self.__disable_https_warnings()
        self.__set_ca_bundle_path()

    def setup_opsgenie_client(self):
        configuration = Configuration(apikey=self.api_key, endpoint=self.apiEndpoint)
        SwaggerConf.ssl_ca_cert = os.environ.get('REQUESTS_CA_BUNDLE')
        configuration.swag_conf = SwaggerConf

        if self.proxy_host:
            proxy_uri = urlparse(self.proxy_host)
            proxy_config = ProxyConfiguration(self.proxy_host, self.proxy_port, self.proxy_username, self.proxy_password, proxy_uri.scheme)
            configuration.proxy_config = proxy_config


    def __disable_https_warnings(self):
        import requests.packages.urllib3
        requests.packages.urllib3.disable_warnings()


    def __extract_file_from_jar(self, config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file = None
            try:
                tmp_file, tmp_abs_path = tempfile.mkstemp()
                Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
                return tmp_abs_path
            finally:
                if tmp_file:
                    tmp_file.close()
        else:
            return None

    def __set_ca_bundle_path(self):
        os.environ['REQUESTS_CA_BUNDLE'] = self.__extract_file_from_jar("certifi/cacert.pem")


    def enable_disable_integration(self, integration_id, action):
        headers = {'content-type': 'application/json', 'Accept': 'application/json', 'Accept-Type': 'application/json',
                   'Authorization': 'GenieKey %s' % (self.api_key)}

        url = self.apiEndpoint + 'integrations/' + integration_id + '/' + action
        try:
            response = requests.post(url, headers=headers)
            if response.status_code == 200:
                print "```Integration disabled.```"
            else:
                print "```Problem while disabling integration. Returned code: %s```" % response.status_code
                sys.exit(1)
        except Exception as er:
            print "```Exception while calling disable integration API.```", er
            sys.exit(1)