#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#
from kubernetes.client.configuration import Configuration
from kubernetes.client.api_client import ApiClient
from kubernetes.client.rest import RESTClientObject
from kubernetes.config.kube_config import KubeConfigLoader
from openshift.dynamic.client import DynamicClient
import base64, os
from xlropenshift.client.common_utils import CommonUtils


class OpenShiftConfigurationObject(Configuration):
    def auth_settings(self):
        if self.get_api_key_with_prefix('authorization'):
            return super(OpenShiftConfigurationObject, self).auth_settings()
        else:
            return {'BearerToken': None}


class OpenShiftClient(object):
    def __init__(self, server):
        self.apiclient = OpenShiftClient.create_api_client(server)
        self.dynclient = DynamicClient(self.apiclient)

    @staticmethod
    def create_api_client(server):
        if server['skipSSL']:
            OpenShiftClient._disable_https_warnings()
        cluster = OpenShiftClient._create_cluster(server_url=server['serverUrl'], cert_path=server['certPath'])
        user = OpenShiftClient._create_user(token=server['token'], tls_cert=server['tlsCert'], tls_private_key=server['tlsPrivateKey'])
        context = OpenShiftClient._create_context()
        config_params = OpenShiftClient._get_config_params(cluster=cluster, context=context, user=user)
        return OpenshiftApiClient(configuration=OpenShiftClient.__get_config(config_params, server['serverUrl'], server['skipSSL']))

    @staticmethod
    def __get_config(config_params, server_url, skip_ssl):
        config = OpenShiftConfigurationObject()
        config.verify_ssl = not skip_ssl
        config_loader = KubeConfigLoader(config_dict=config_params)
        config_loader.host = server_url
        config_loader.load_and_set(client_configuration=config)
        if not config.ssl_ca_cert:
            config.ssl_ca_cert = OpenShiftClient._set_ca_bundle_path(path="certifi/cacert.pem")
        return config

    @staticmethod
    def _create_cluster(server_url=None, cert_path=None):
        cluster = {"server": server_url,
                   "certificate-authority-data": base64.b64encode(
                       CommonUtils.remove_certification_strip(cert_path)) if cert_path else None}
        return CommonUtils.remove_none_keys(cluster)

    @staticmethod
    def _create_context():
        return {"cluster": "kb_cluster", "user": "kb_user"}

    @staticmethod
    def _create_user(token=None, tls_cert=None, tls_private_key=None):
        user = {
            "client-certificate-data": base64.b64encode(
                CommonUtils.remove_certification_strip(tls_cert)) if tls_cert else None,
            "client-key-data": base64.b64encode(
                CommonUtils.remove_certification_strip(tls_private_key)) if tls_private_key else None,
            "token": token}
        return CommonUtils.remove_none_keys(user)

    @staticmethod
    def _get_config_params(context, cluster, user):
        return {"current-context": "kb_context",
                "contexts": [{"name": "kb_context", "context": context}],
                "clusters": [{"name": context['cluster'], "cluster": cluster}],
                "users": [{"name": context['user'], "user": user}]}

    @staticmethod
    def _disable_https_warnings():
        import urllib3
        urllib3.disable_warnings()

    @staticmethod
    def _set_ca_bundle_path(path):
        ca_bundle_path = CommonUtils.extract_file_from_jar(path)
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path
        return ca_bundle_path

    def client(self):
        return self.apiclient

    def dyn_client(self):
        return self.dynclient

class OpenshiftApiClient(ApiClient):
    def __init__(self, configuration=None, header_name=None, header_value=None, cookie=None):
        if configuration is None:
            configuration = OpenShiftConfigurationObject()
        self.configuration = configuration
        self.rest_client = RESTClientObject(configuration)
        self.default_headers = {}
        if header_name is not None:
            self.default_headers[header_name] = header_value
        self.cookie = cookie
        # Set default User-Agent.
        self.user_agent = 'Swagger-Codegen/10.0.1/python'