#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import com.xhaus.jyson.JysonCodec as json
import sets
import sys
import time
from xlrelease.HttpRequest import HttpRequest

HTTP_SUCCESS = sets.Set([200, 201])


class OctopusClient(object):

    # HTTP Connection

    def __init__(self, httpConnection, apiKey):
        self.httpConnection = httpConnection
        self.httpRequest = HttpRequest(httpConnection)
        self.apiKey = apiKey
        self.headers = self._get_headers()

    @staticmethod
    def create_client(httpConnection, apiKey):
        return OctopusClient(httpConnection, apiKey)

    def _get_headers(self):
        return {"Accept": "application/json", "Content-Type": "application/json", "X-Octopus-ApiKey": self.apiKey}

    def httpGetUrl(self, url):
        response = self.httpRequest.get(url, headers=self.headers)
        if response.getStatus() not in HTTP_SUCCESS:
            sys.exit(
                "Error from server, HTTP Return: %s, content %s\n" % (response.getStatus(), response.getResponse()))
        data = json.loads(response.getResponse())
        return data

    def httpPostUrl(self, url, body):
        response = self.httpRequest.post(url, headers=self.headers, body=body)
        print("HTTP_STATUS = %s" % response.getStatus())
        if response.getStatus() in HTTP_SUCCESS:
            data = json.loads(response.getResponse())
            return data["Id"]
        sys.exit("Error from server, HTTP Return: %s, content %s\n" % (response.getStatus(), response.getResponse()))

    def ping(self):
        print(json.dumps(self.httpGetUrl('/api/serverstatus')))

    # Functions to get IDs

    def get_project_id(self, project):
        '''Gets the technical ID of an Octopus project (a software project with a deployment process and
        environment on Octopus Deploy). Users no longer need this to create releases and deploy - here for
        backwards compatibility.'''
        data = self.httpGetUrl('/api/projects/%s' % project)
        projectId = data["Id"]
        print "Found Project Id [%s] for name [%s]" % (projectId, project)
        return projectId

    def get_release_id(self, projectName, releaseName):
        '''Gets the technical ID of an Octopus release (the release process for a project).
        Helper function for starting Octopus deployments.'''
        projectId = self.get_project_id(projectName)
        url = '/api/projects/%s/releases/%s' % (projectId, releaseName)
        data = self.httpGetUrl(url)
        releaseId = data["Id"]
        print "Found Release Id [%s] for name [%s]" % (releaseId, releaseName)
        return releaseId

    def get_environment_id(self, environment):
        '''Gets the technical ID of an Octopus environment (a deployment target).
         Users no longer need this to create releases and deploy - here for backwards compatibility.'''
        data = self.httpGetUrl('/api/environments/all')
        for obj in data:
            if obj["Name"] == environment:
                print "Found Environment Id [%s] for name [%s]" % (obj["Id"], environment)
                return obj["Id"]
        sys.exit("Not Found")

    def getChannelId(self, channel, projectId):
        '''Gets the technical ID of an Octopus channel (channels are versions of a project with different release
        processes). Helper function for creating releases.'''
        data = self.httpGetUrl('api/channels')
        print("data = %s" % data)
        for obj in data["Items"]:
            print "NAME = %s ID = %s" % (obj["Name"], obj["Id"])
            if obj["Name"] == channel and obj["ProjectId"] == projectId:
                return obj["Id"]
        sys.exit("Not Found")

    # Helper function for deployments

    def wait_for_deploy(self, deploymentId):
        '''Waits for an Octopus deployment to finish and returns the result (success or failure)'''
        deployment_details = self.httpGetUrl('/api/deployments/%s' % deploymentId)
        taskUrl = deployment_details["Links"]["Task"]
        time.sleep(5)
        task_details = self.httpGetUrl(taskUrl)
        while not task_details["IsCompleted"]:
            task_details = self.httpGetUrl(taskUrl)
            time.sleep(5)
        if task_details["FinishedSuccessfully"]:
            print("Deployment finished successfully.")
        else:
            sys.exit("Deployment failed, errors: [%s]" % task_details["ErrorMessage"])

    # Start deployments

    def start_deploy(self, projectName, releaseName, environmentName):
        '''Starts an octopus deployment, using get_release_id and get_environment_id as helper functions'''
        releaseId = self.get_release_id(projectName, releaseName)
        environmentId = self.get_environment_id(environmentName)
        url = '/api/deployments'
        data = {
            "ReleaseId": releaseId,
            "EnvironmentId": environmentId,
            "TenantId": None,
            "SkipActions": [],
            "QueueTime": None,
            "QueueTimeExpiry": None,
            "FormValues": {},
            "ForcePackageDownload": False,
            "UseGuidedFailure": False,
            "SpecificMachineIds": [],
            "ExcludedMachineIds": [],
            "ForcePackageRedeployment": False
        }
        print("data = %s" % data)
        return self.httpPostUrl(url, json.dumps(data))

    # Get package information to create releases

    def getProjectPackageInformation(self, projectId):
        '''Gets version information about packages in a project to release - originally this function was created
        by John Hancock and meant for GUI users - now this is used directly in create_release as a helper function.
        Still available to users for backward compatibility.'''
        data = self.httpGetUrl('/api/deploymentprocesses/deploymentprocess-%s' % projectId)
        print("data = %s" % data)
        projectPackages = ""
        for obj in data["Steps"]:
            for act in obj["Actions"]:
                if act["ActionType"] == "Octopus.TentaclePackage":
                    if not projectPackages == "":
                        projectPackages = "%s," % projectPackages
                    stepName = act["Name"]
                    packageId = act["Properties"]["Octopus.Action.Package.PackageId"]
                    packageVersion = self.getPackageVersion(packageId)
                    packageEntry = '{"StepName":"%s","Version":"%s"}' % (stepName, packageVersion)
                    print ("PACKAGE ENTRY - %s" % packageEntry)
                    projectPackages = "%s%s" % (projectPackages, packageEntry)
        return "[%s]" % projectPackages

    def getPackageVersion(self, packageId):
        '''Finds the version of a specific package - helper function for getProjectPackageInformation'''
        data = self.httpGetUrl('api/packages?skip=0&take=1000')
        print("data = %s" % data)
        print("Searching for %s" % packageId)
        for obj in data["Items"]:
            if obj["PackageId"].lower() == packageId.lower():
                print("Version for %s is %s" % (packageId, obj["Version"]))
                return obj["Version"]

    # Increment version number for releases

    def getNextVersion(self, projectId):
        '''Increments the current release's version number to create a new version number for a new release.
        originally this function was created by John Hancock and meant for GUI users - now this is used directly in
         create_release as a helper function. Still available to users for backward compatibility.'''
        data = self.httpGetUrl('api/projects/%s/releases' % projectId)
        print("data = %s" % data)
        for obj in data["Items"]:
            print "Most recent RELEASE ID = %s, VERSION = %s" % (obj["Id"], obj["Version"])
            currentVersion = obj["Version"]
            break
        if currentVersion == "":
            sys.exit("Not Found")
        # Logic is based on a #.#.# Octopus Versioning standard
        currentVer = self.formatOctopusVersion(currentVersion)
        ver = currentVer.split('.')
        nextVer = int(ver[len(ver) - 1]) + 1
        print "Next release version: %s.%s.%s" % (ver[0], ver[1], nextVer)
        return "%s.%s.%s" % (ver[0], ver[1], nextVer)

    def formatOctopusVersion(self, version):
        '''Helper function for getNextVersion'''
        ver = version.split('.')
        if len(ver) < 3:
            return self.formatOctopusVersion("%s.0" % version)
        return version

    # Create a release

    def create_release(self, project, version, releaseNotes, selectedPackages, channel):
        '''Creates a release on Octopus Deploy, using helper functions to get project id, channel id, package
        versions and version number as needed'''
        projectId = self.get_project_id(project)
        channelId = self.getChannelId(channel, projectId)
        url = '/api/releases'
        if selectedPackages == "":
            selectedPackages = self.getProjectPackageInformation(projectId)
            packages = json.loads(selectedPackages)
        else:
            packages = json.loads(selectedPackages)
        if version == "":
            version = self.getNextVersion(projectId)
        data = {
            "Version": version,
            "ProjectId": projectId,
            "ChannelId": channelId,
            "SelectedPackages": packages,
            "ReleaseNotes": releaseNotes
        }
        return (self.httpPostUrl(url, json.dumps(data)), version)
