#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import com.xhaus.jyson.JysonCodec as json
import sets
import sys
import re
import time
from octopus import create_record
from xlrelease.HttpRequest import HttpRequest
import urllib

HTTP_SUCCESS = sets.Set([200, 201])


class OctopusClient(object):

    # HTTP Connection

    def __init__(self, httpConnection, apiKey):
        self.httpConnection = httpConnection
        self.httpRequest = HttpRequest(httpConnection)
        self.apiKey = apiKey
        self.headers = self._get_headers()

    @staticmethod
    def create_client(httpConnection, apiKey):
        return OctopusClient(httpConnection, apiKey)

    def _get_headers(self):
        return {"Accept": "application/json", "Content-Type": "application/json", "X-Octopus-ApiKey": self.apiKey}

    def httpGetUrl(self, url):
        response = self.httpRequest.get(url, headers=self.headers)
        if response.getStatus() not in HTTP_SUCCESS:
            sys.exit(
                "Error from server, HTTP Return: %s, content %s" % (response.getStatus(), response.getResponse()))
        data = json.loads(response.getResponse())
        return data

    def httpPostUrl(self, url, body):
        response = self.httpRequest.post(url, headers=self.headers, body=body)
        if response.getStatus() in HTTP_SUCCESS:
            data = json.loads(response.getResponse())
            return data, response.getStatus()
        sys.exit("Error from server, HTTP Return: %s, content %s" % (response.getStatus(), response.getResponse()))

    def ping(self):
        print(json.dumps(self.httpGetUrl('/api/serverstatus')))

    # Functions to get IDs
    def get_all_projects(self):
        return self.httpGetUrl('/api/projects/all')

    def get_project_info(self, project):
        data = self.get_all_projects()
        for obj in data:
            if obj["Name"] == project:
                print("Found project details %s for project name %s." % (obj, project))
                return (obj["Id"], obj["LifecycleId"], obj["SpaceId"], obj["ProjectGroupId"], project)
        sys.exit("No project found with name %s" % (project))

    def get_project_id(self, project):
        '''Gets the technical ID of an Octopus project (a software project with a deployment process and
        env on Octopus Deploy). Users no longer need this to create releases and deploy - here for
        backwards compatibility.'''
        print("Finding the project ID for %s:" % project)
        data = self.get_all_projects()
        for obj in data:
            if obj["Name"] == project:
                print("Found project ID %s for name %s." % (obj["Id"], project))
                return (obj["Id"])
        sys.exit("No project found with name %s" % (project))


    def get_release_id(self, projName, relName):
        '''Gets the technical ID of an Octopus release (the release process for a project).
        Helper function for starting Octopus deployments.'''
        print("Finding the release ID for project %s and release %s:" % (projName, relName))
        projId = self.get_project_id(projName)
        url = '/api/projects/%s/releases/%s' % (projId, relName)
        data = self.httpGetUrl(url)
        relId = data["Id"]
        print "Found Release Id %s for name %s." % (relId, relName)
        return relId

    def get_all_environments(self):
        data = self.httpGetUrl('/api/environments/all')
        return data

    def get_environment_name(self, id):
        print("Finding the environment Name for environment id %s:" % id)
        data = self.get_all_environments()
        for obj in data:
            if obj["Id"] == id:
                print "Found environment Name %s for id %s." % (obj["Name"], id)
                return obj["Name"]
        sys.exit("Not Found")

    def get_environment_id(self, env):
        '''Gets the technical ID of an Octopus env (a deployment target).
             Users no longer need this to create releases and deploy - here for backwards compatibility.'''
        print("Finding the environment ID for environment %s:" % env)
        data = self.httpGetUrl('/api/environments/all')
        for obj in data:
            if obj["Name"] == env:
                print "Found environment Id %s for name %s." % (obj["Id"], env)
                return obj["Id"]
        sys.exit("Not Found")

    def getChannelId(self, channel, projId):
        '''Gets the technical ID of an Octopus channel (channels are versions of a project with different release
        processes). Helper function for creating releases.'''
        print("Finding the channel ID for channel %s and project ID %s:" % (channel, projId))
        data = self.httpGetUrl('api/projects/%s/channels' % projId)
        print("API response for debugging purposes: \n\n"+str(data)+"\n\n")
        for obj in data["Items"]:
            if obj["Name"] == channel:
                print("Found channel ID %s." % obj["Id"])
                return obj["Id"]
        sys.exit("Not Found")

    def get_lifecyle(self, lifecycleId):
        print("Finding the lifecycles for id %s. Starting dynamic generation" % lifecycleId)
        return self.httpGetUrl('/api/lifecycles/%s' % lifecycleId)

    def abort_task(self, taskId):
        return self.httpPostUrl('/api/tasks/'+taskId+'/cancel', '')

    # Start deployments
    def start_deploy(self, projName, relName, envName, taskReportingApi, task, server):
        '''Starts an octopus deployment, using get_release_id and get_environment_id as helper functions'''
        relId = self.get_release_id(projName, relName)
        envId = self.get_environment_id(envName)
        url = '/api/deployments'
        data = {
            "ReleaseId": relId,
            "EnvironmentId": envId,
            "TenantId": None,
            "SkipActions": [],
            "QueueTime": None,
            "QueueTimeExpiry": None,
            "FormValues": {},
            "ForcePackageDownload": False,
            "UseGuidedFailure": False,
            "SpecificMachineIds": [],
            "ExcludedMachineIds": [],
            "ForcePackageRedeployment": False
        }

        resp,status = self.httpPostUrl(url, json.dumps(data))

        project_url = resp["Links"]["Project"]

        if status in HTTP_SUCCESS:
            status = "completed"
        else:
            status = "failed"

        create_record(taskReportingApi, task, server, status=status)
        return resp["Id"], resp['TaskId']

    # Get package information to create releases

    def getProjectPackageInformation(self, projId, spaceId=None, gitRef=None):
        '''Gets version information about packages in a project to release - originally this function was created
        by John Hancock and meant for GUI users - now this is used directly in create_release as a helper function.
        Still available to users for backward compatibility.'''
        print("Finding complete package information for project with ID %s:" % projId)
        
        if spaceId and gitRef:
            git_ref_encoded = urllib.quote(gitRef, safe='')
            deployment_process_url = '/api/%s/projects/%s/%s/deploymentprocesses' % (spaceId, projId, git_ref_encoded)
        else:
            deployment_process_url = '/api/deploymentprocesses/deploymentprocess-%s' % projId
        
        data = self.httpGetUrl(deployment_process_url)
        projectPackages = ""
        for obj in data["Steps"]:
            for act in obj["Actions"]:
                if act["Properties"] and "Octopus.Action.Package.PackageId" in act["Properties"]:
                    if not projectPackages == "":
                        projectPackages = "%s," % projectPackages
                    stepName = act["Name"]
                    packageId = act["Properties"]["Octopus.Action.Package.PackageId"]
                    packageVersion = self.getPackageVersion(packageId)
                    packageEntry = '{"StepName":"%s","Version":"%s"}' % (stepName, packageVersion)
                    projectPackages = "%s%s" % (projectPackages, packageEntry)
        print("Package information is: [%s]." % projectPackages)
        return "[%s]" % projectPackages

    def getPackageVersion(self, packageId):
        '''Finds the version of a specific package - helper function for getProjectPackageInformation'''
        print("Finding version information for package %s:" % packageId)
        data = self.httpGetUrl('api/packages?skip=0&take=1000')
        print("Searching for %s" % packageId)
        for obj in data["Items"]:
            if obj["PackageId"].lower() == packageId.lower():
                print("Version for %s is %s" % (packageId, obj["Version"]))
                return obj["Version"]

    # Increment version number for releases

    def getNextVersion(self, projId):
        '''Increments the current release's version number to create a new version number for a new release.
        originally this function was created by John Hancock and meant for GUI users - now this is used directly in
         create_release as a helper function. Still available to users for backward compatibility.'''
        print("Creating new release version number based on the most recent release version number:")
        data = self.httpGetUrl('api/projects/%s/releases' % projId)
        for obj in data["Items"]:
            print "Most recent release ID is %s, version is %s." % (obj["Id"], obj["Version"])
            currentVersion = obj["Version"]
            break
        if currentVersion == "":
            sys.exit("Not Found")

        # Logic is based on a #.#.# Octopus Versioning standard
        currentVer = self.formatOctopusVersion(currentVersion)
        ver = currentVer.split('.')
        patchVersion = ver[2]
        uniPatchVersion = unicode(patchVersion, 'utf-8')
        result = ""

        if (uniPatchVersion.isnumeric()):
            #---- test strings below ----
            #1.0.1 > 1.0.2
            nextVer = int(ver[len(ver) - 1]) + 1
            result = str(ver[0]) + "." + str(ver[1]) + "." + str(nextVer)
            print "Next release version is --  : %s" % result
        else:
            #---- test strings below ----
            #1.0.0-alpha.beta > 1.0.1-alpha.beta
            #1.0.5-alpha.beta > 1.0.6-alpha.beta
            #1.2.3-c > 1.2.4-c
            #2016.09.01-beta.0001 > 2016.09.2-beta.0001
            #2.156.190905-154c81a-t59 > 2.156.190906-154c81a-t59

            alphaNum = currentVersion.split('-')
            desc = ""
            for item in range(len(alphaNum) - 1):
                desc += "-" + alphaNum[item + 1]

            p1 = patchVersion.split('-')
            unicodeP1 = unicode(p1[0], 'utf-8')
            if (unicodeP1.isnumeric()):
                nextVer = int(unicodeP1) + 1
                result = str(ver[0]) + "." + str(ver[1]) + "." + str(nextVer) + desc
                print "Next release version is --  : %s " % result
            else:
                #---- test strings below ----
                #2.156190905-154c81a-t59.uu-22 > 2.156190905.1
                print "patch version invalid. Expected a number"
                minor = ver[1].split('-')
                result = str(ver[0]) + "." + str(minor[0]) + "." + str(1)
                print "Next release version is --  : %s " % result
        return result

    def formatOctopusVersion(self, newVersion):
        '''Helper function for getNextVersion'''
        ver = newVersion.split('.')
        if len(ver) < 3:
            return self.formatOctopusVersion("%s.0" % newVersion)
        return newVersion

    # Create a release

    def create_release(self, project, version, releaseNotes, selectPackages, channel, taskReportingApi, task, server, gitRef):
        '''Creates a release on Octopus Deploy, using helper functions to get project id, channel id, package
        versions and version number as needed'''
        projId, lifecycleId, spaceId, projectGroupId, projectName = self.get_project_info(project)
        data = {}
        if gitRef:
            data.update({
                "VersionControlReference": {
                    'GitRef':gitRef
                    }})
        channelId = self.getChannelId(channel, projId)
        url = '/api/releases'
        if selectPackages == "":
            selectPackages = self.getProjectPackageInformation(projId, spaceId, gitRef)
            packages = json.loads(selectPackages)
        else:
            packages = json.loads(selectPackages)
        if version == "":
            newVersion = self.getNextVersion(projId)
        else:
            newVersion = version
        data.update({
            "Version": newVersion,
            "ProjectId": projId,
            "ChannelId": channelId,
            "ReleaseNotes": releaseNotes
        })
        
        if packages:
            data["SelectedPackages"] = packages

        print("Creating release for project ID %s, version %s, release notes '%s', packages %s, channel ID %s." %
              (projId, newVersion, releaseNotes, packages, channelId))
        resp, status = self.httpPostUrl(url, json.dumps(data))

        if status in HTTP_SUCCESS:
            status = "completed"
        else:
            status = "failed"
        create_record(taskReportingApi, task, server, status=status)

        return (resp["Id"], newVersion)
