#
# Copyright 2025 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import com.xhaus.jyson.JysonCodec as json
import urllib
from xlrelease.HttpRequest import HttpRequest
from xlrelease.OAuthSupport import OAuthHttpClient
from msteams.util import error
from msteams.util import validate_mandatory_field

class TeamsServer(object):
    def __init__(self, httpConnection):
        validate_mandatory_field(httpConnection, "MS Teams Server not configured")
        self.httpRequest = HttpRequest(httpConnection)
        self.headers={"Accept": "application/json", "Content-Type": "application/json"}
        access_token = TeamsServer.get_access_token(httpConnection)
        self.headers['Authorization'] = "Bearer %s" % access_token

    @staticmethod
    def create_client(httpConnection):
        access_token_url = "https://login.microsoftonline.com/%s/oauth2/v2.0/token" % httpConnection['tenantId']
        httpConnection['accessTokenUrl'] = access_token_url
        return TeamsServer(httpConnection)

    @staticmethod
    def get_access_token(httpConnection):
        body = dict(
            grant_type = "password",
            client_id = httpConnection['clientId'],
            client_secret = httpConnection['clientSecret'],
            username = httpConnection['username'],
            password = httpConnection['password'],
            scope = httpConnection['scope']
        )
        oauth_client = OAuthHttpClient(httpConnection)
        response = oauth_client.post(httpConnection['accessTokenUrl'], body=urllib.urlencode(body))
        response_body = json.loads(response.getResponse())
        if response.getStatus() == 200:
           return response_body['access_token']
        else:
           error("", response)

    def list_teams(self, team_name):
        url = "/teams?$filter=startswith(displayName, '%s')" % team_name
        response = self.httpRequest.get(url, headers=self.headers)
        if response.status != 200:
            return []
        data = json.loads(response.getResponse())
        teams = data["value"]
        return teams

    def list_channels(self, team_id, channel_name):
        url = "/teams/%s/channels?$filter=startswith(displayName, '%s')" % (team_id, channel_name)
        response = self.httpRequest.get(url, headers=self.headers)
        if response.status != 200:
            return []
        data = json.loads(response.getResponse())
        list_channels = data["value"]
        return list_channels

    def send_message_to_channel(self, team_id, channel_id, message):
        url = "/teams/%s/channels/%s/messages" % (team_id, channel_id)
        payload={
            "body": {
                "content": message
            }
        }
        response = self.httpRequest.post(url, json.dumps(payload), headers=self.headers)
        if response.status != 201:
           error("An unexpected error occurred while sending the message to the channel", response)
        response_body = json.loads(response.getResponse())
        return response_body

    def get_authorized_user_id(self):
        response = self.httpRequest.get("/me", headers=self.headers)
        if response.status != 200:
           error("An unexpected error occurred while fetching the authorized user ID", response)
        data=json.loads(response.getResponse())
        user_id = data["id"]
        return user_id

    def list_users(self, email_id):
        url = "/users?$filter=startswith(mail, '%s')" % email_id
        response = self.httpRequest.get(url, headers=self.headers)
        if response.status != 200:
            return []
        data = json.loads(response.getResponse())
        users = data["value"]
        return users

    def get_user_id(self, user_email):
        url="/users/%s" % user_email
        response = self.httpRequest.get(url, headers=self.headers)
        if response.status != 200:
            error("An unexpected error occurred while fetching the user ID", response)
        data = json.loads(response.getResponse())
        user_id = data["id"]
        return user_id

    def create_user_chat(self, authorized_user_id, user_id):
        payload = {
            "chatType": "oneOnOne",
            "members": [
                {
                    "@odata.type": "#microsoft.graph.aadUserConversationMember",
                    "roles": ["owner"],
                    "user@odata.bind": "https://graph.microsoft.com/v1.0/users('{}')".format(authorized_user_id)
                },
                {
                    "@odata.type": "#microsoft.graph.aadUserConversationMember",
                    "roles": ["owner"],
                    "user@odata.bind": "https://graph.microsoft.com/v1.0/users('{}')".format(user_id)
                }
            ]
        }
        response = self.httpRequest.post('/chats', json.dumps(payload), headers=self.headers)
        if response.status != 201:
            error("An unexpected error occurred while creating the user chat", response)
        response_body = json.loads(response.getResponse())
        chat_id = response_body["id"]
        return chat_id

    def get_group_list(self, group_name):
        url = "/me/chats?$filter=chatType eq 'group' and startswith(topic, '%s')" % group_name
        response = self.httpRequest.get(url, headers=self.headers)
        if response.status != 200:
            return []
        data = json.loads(response.getResponse())
        group_list = data["value"]
        return group_list

    def send_message_to_chat(self, chat_id, message):
        url = "/chats/%s/messages" % chat_id
        payload={
            "body": {
                "content": message
            }
        }
        response = self.httpRequest.post(url, json.dumps(payload), headers=self.headers)
        if response.status != 201:
            error("An unexpected error occurred while sending the message to the chat", response)
        response_body = json.loads(response.getResponse())
        return response_body