'use strict';

(function () {

    var JiraQueryTileViewController = function JiraQueryTileViewController($scope, JiraQueryService, XlrTileHelper) {
        var vm = this;

        vm.tileConfigurationIsPopulated = tileConfigurationIsPopulated;

        var tile;

        var predefinedColors = [];
        predefinedColors['Open'] = '#7E827A';
        predefinedColors['Reopened'] = '#4AA0C8';
        predefinedColors['To Do'] = '#7E827A';
        predefinedColors['In Progress'] = '#7FB2F0';
        predefinedColors['Resolved'] = '#45BF55';
        predefinedColors['Done'] = '#45BF55';
        predefinedColors['Closed'] = '#468966';
        predefinedColors['Testing'] = '#FFA500';

        var colorPool = ['#7E8AA2', '#35203B', '#644D52', '#8E2800', '#B85C5A', '#FF8598', '#FF6F69', '#F77A52', '#FCD364', '#FFE11A', '#FFA500'];

        if ($scope.xlrTile) {
            // summary mode
            tile = $scope.xlrTile.tile;
        } else {
            // details mode
            tile = $scope.xlrTileDetailsCtrl.tile;
        }

        function tileConfigurationIsPopulated() {
            var config = tile.configurationProperties;
            return !_.isEmpty(config.jiraServer) && !_.isEmpty(config.query);
        }

        function getColor(value) {
            if (predefinedColors[value]) return predefinedColors[value];
            return colorPool.pop();
        }

        vm.chartOptions = {
            topTitleText: function topTitleText(data) {
                return data.total;
            },
            bottomTitleText: 'issues',
            series: function series(data) {
                var series = {
                    name: 'Status',
                    data: []
                };
                series.data = _.map(data.data, function (value) {
                    return { y: value.counter, name: value.status, color: value.color };
                });
                return [series];
            },
            showLegend: false,
            donutThickness: '60%'
        };

        function load(config) {
            if (tileConfigurationIsPopulated()) {
                vm.loading = true;
                JiraQueryService.executeQuery(tile.id, config).then(function (response) {
                    var jiraIssueArray = [];
                    var issues = response.data.data;
                    vm.statuses = [];
                    vm.statusesCounter = 0;
                    vm.issuesSummaryData = {
                        data: null,
                        total: 0
                    };
                    vm.issuesSummaryData.data = _.reduce(issues, function (result, value) {
                        var status = value.status;
                        vm.issuesSummaryData.total += 1;
                        if (result[status]) {
                            result[status].counter += 1;
                        } else {
                            result[status] = {
                                counter: 1,
                                color: getColor(status),
                                status: status
                            };
                        }
                        value.color = result[status].color;
                        jiraIssueArray.push(value);
                        return result;
                    }, {});
                    _.forEach(vm.issuesSummaryData.data, function (value, key) {
                        if (vm.statusesCounter < 5) vm.statuses.push(value);
                        vm.statusesCounter++;
                    });
                    vm.gridOptions = createGridOptions(jiraIssueArray);
                }).finally(function () {
                    vm.loading = false;
                });
            }
        }

        function createGridOptions(jiraData) {
            var filterHeaderTemplate = "<div data-ng-include=\"'partials/releases/grid/templates/name-filter-template.html'\"></div>";
            var columnDefs = [{
                displayName: "Issue",
                field: "issue",
                cellTemplate: "static/5.0.0/include/JiraQueryTile/grid/issue-cell-template.html",
                filterHeaderTemplate: filterHeaderTemplate,
                enableColumnMenu: false,
                width: '25%'
            }, {
                displayName: "Summary",
                field: "summary",
                cellTemplate: "static/5.0.0/include/JiraQueryTile/grid/summary-cell-template.html",
                filterHeaderTemplate: filterHeaderTemplate,
                enableColumnMenu: false,
                width: '25%'
            }, {
                displayName: "Status",
                field: "status",
                cellTemplate: "static/5.0.0/include/JiraQueryTile/grid/status-cell-template.html",
                filterHeaderTemplate: filterHeaderTemplate,
                enableColumnMenu: false,
                width: '25%'
            }, {
                displayName: "Assigned",
                field: "assignee",
                cellTemplate: "static/5.0.0/include/JiraQueryTile/grid/assigned-cell-template.html",
                filterHeaderTemplate: filterHeaderTemplate,
                enableColumnMenu: false,
                width: '25%'
            }];

            return XlrTileHelper.getGridOptions(jiraData, columnDefs);
        }

        function refresh() {
            load({ params: { refresh: true } });
        }

        load();

        vm.refresh = refresh;
    };

    JiraQueryTileViewController.$inject = ['$scope', 'xlrelease.jira.JiraQueryService', 'XlrTileHelper'];

    var JiraQueryService = function JiraQueryService(Backend) {

        function executeQuery(tileId, config) {
            return Backend.get("tiles/" + tileId + "/data", config);
        }

        return {
            executeQuery: executeQuery
        };
    };

    JiraQueryService.$inject = ['Backend'];

    angular.module('xlrelease.jira.tile', []);
    angular.module('xlrelease.jira.tile').service('xlrelease.jira.JiraQueryService', JiraQueryService);
    angular.module('xlrelease.jira.tile').controller('jira.JiraQueryTileViewController', JiraQueryTileViewController);
})();