import com.xhaus.jyson.JysonCodec as Json
import string
import sys
from base64 import b64encode
from java.time import Instant
from java.util import Date
from java.text import SimpleDateFormat

from util import error
from xlrelease.HttpRequest import HttpRequest
from java.time.format import DateTimeFormatter

# Constants
MAX_RESULTS = 1000

class JiraServer:

    def __init__(self, jira_server, username, password, api_token, pat, task_reporting_api = None, task = None):
        if jira_server is None:
            error('No server provided.')

        self.jira_server = jira_server
        self.username = username
        self.password = password
        self.api_token = api_token
        self.pat = pat
        self.task_reporting_api = task_reporting_api
        self.task = task
        self.hosting_option = self._getHostingOption()

    def queryIssues(self, query, options = None):
        if not query:
            error('No JQL query provided.')

        # Collect issue data based on hosting type
        request_body = {
            'jql': query,
            'fields': ['summary', 'status', 'assignee']
        }

        if self._getHostingOption() == 'CLOUD':
            query_data = self._executeCloudSearch(request_body)
        else:
            query_data = self._executeServerSearch(request_body)

        issues = {}
        for item in query_data:
            issue = item['key']
            assignee = "Unassigned" if (item['fields']['assignee'] is None) else item['fields']['assignee']['displayName']

            issues[issue] = {
                'issue'   : issue,
                'summary' : item['fields']['summary'],
                'status'  : item['fields']['status']['name'],
                'assignee': assignee,
                'link'    : "{1}/browse/{0}".format(issue, self.jira_server['url'])
            }

        return issues

    def query(self, query, quiet = False, getKey = False):
        if not query:
            error('No JQL query provided.')

        # Collect issue data based on hosting type
        request_body = {
            'jql': query,
            'fields': ['summary', 'issuetype', 'updated', 'status', 'reporter']
        }

        if self._getHostingOption() == 'CLOUD':
            # Cloud /search/jql endpoint doesn't support expand parameter
            issue_data = self._executeCloudSearch(request_body, max_results=100)
        else:
            request_body['expand'] = ['changelog']
            issue_data = self._executeServerSearch(request_body)

        # Process the collected data
        if not quiet and len(issue_data) > 0:
            print "#### Issues found"

        issues = {}
        issueKeys = []

        for item in issue_data:
            issue = item['key']
            issueKeys.append(issue)
            issues[issue] = item['fields']['summary']
            if not quiet:
                print(u"* {0} - {1}".format(self.link(issue), item['fields']['summary']))

            try:
                if self.task_reporting_api and self.task:
                    planRecord = self.task_reporting_api.newPlanRecord()
                    planRecord.targetId = self.task.id
                    planRecord.ticket = issue
                    planRecord.title = item['fields']['summary']
                    planRecord.ticketType = (item['fields'].get('issuetype', {})).get('name')
                    updated = item['fields'].get('updated')
                    if updated:
                        date_formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSX")
                        planRecord.updatedDate = Date.from(Instant.from(date_formatter.parse(updated)))

                    histories = item.get('changelog', {}).get('histories', [])
                    if len(histories) > 0:
                        updatedBy = next(iter(histories), {}).get('author', {}).get('displayName')
                    else:
                        updatedBy = item['fields'].get('reporter', {}).get('displayName')

                    if updatedBy:
                        planRecord.updatedBy = updatedBy

                    status = item['fields'].get('status', {}).get('name')
                    if status:
                        planRecord.status = status
                    planRecord.ticket_url = self.jira_server['url'].rstrip('/') + '/browse/' + issue
                    planRecord.serverUrl = self.jira_server['url']
                    if not self.username:
                        planRecord.serverUser = self.jira_server['username']
                    else:
                        planRecord.serverUser = self.username

                    self.task_reporting_api.addRecord(planRecord, True)
            except:
                print "\n\n"
                print sys.exc_info()

        if not quiet and len(issue_data) == 0:
            print "#### No issues found"
        if not quiet:
            print "\n"
        if getKey:
            return issues, issueKeys
        else:
            return issues

    def createIssue(self, project, title, description, issue_type):
        # Create POST body
        content = {
            'fields': {
                'project': {
                    'key': project
                },
                'summary': title,
                'description': description,
                'issuetype': {
                    'name': issue_type
                }
            }
        }

        # Do request
        request = self._createRequest()
        # Api for get project type
        response = request.get(self._getVersionUri() + "/project/%s/" % project,
                               contentType='application/json', headers=self._createApiTokenHeader())

        if response.status == 200:
            if "epic" == issue_type.lower() and "classic" == (Json.loads(response.response))['style']:
                # Get field Id for issue type
                response = request.get(self._getVersionUri() + '/field',
                                   contentType='application/json', headers=self._createApiTokenHeader())
                field_id = ""
                if response.status == 200:
                    for field in Json.loads(response.response):
                        if field['name'] == "Epic Name":
                            field_id = field['id']
                            break

                else:
                    error("Failed to create issue in JIRA.", response)

                # Update field Id for Epic
                content['fields'].update({
                    field_id : title
                })

        else:
            error("Failed to create issue in JIRA.", response)

        response = request.post(self._getVersionUri() + '/issue', self._serialize(content),
                                contentType='application/json', headers=self._createApiTokenHeader())

        # Parse result
        if response.status == 201:
            data = Json.loads(response.response)
            issue_id = data.get('key')
            print u"Created {0} in JIRA.".format(self.link(issue_id))
            self._createPlanRecord(issue_id)
            return issue_id
        else:
            error("Failed to create issue in JIRA.", response)

    def createIssueJson(self, json_obj):
        # Do request
        request = self._createRequest()
        response = request.post(self._getVersionUri() + '/issue', json_obj, contentType='application/json',
                                headers=self._createApiTokenHeader())

        # Parse result
        if response.status == 201:
            data = Json.loads(response.response)
            issue_id = data.get('key')
            print u"Created {0} in JIRA.".format(self.link(issue_id))
            self._createPlanRecord(issue_id)
            return issue_id
        else:
            error("Failed to create issue in JIRA.", response)

    def createSubtask(self, project, title, description, parent_issue_key, issueTypeName):
        # Send an API request to find the projectId
        projectUri = self._getVersionUri() + "/project/%s" % project
        projectRequest = self._createRequest()
        projectResponse = projectRequest.get(projectUri, contentType="application/json",
                                             headers=self._createApiTokenHeader())
                                             
        if projectResponse.status == 200:
            data = Json.loads(projectResponse.response)
            for issueType in data['issueTypes']:
                if issueType['subtask']:
                    if not issueTypeName or issueType['name'].lower() == issueTypeName.lower():
                        type_id = issueType['id']
                        break
            else:
                error("Failed to find issue type '%s' for '%s' project." % (issueTypeName, project))
        else:
            error("Failed to get issue types for '%s' project." % project, issueTypeResponse)
        # Create POST body
        content = {
            'fields': {
                'project': {
                    'key': project
                },
                'parent': {
                    'key': parent_issue_key
                },
                'summary': title,
                'description': description,
                'issuetype': {
                    'id': type_id
                }
            }
        }
        # Do request
        request = self._createRequest()
        response = request.post(self._getVersionUri() + '/issue', self._serialize(content),
                                contentType='application/json', headers=self._createApiTokenHeader())

        # Parse result
        if response.status == 201:
            data = Json.loads(response.response)
            subtask_id = data.get('key')
            print u"Created {0} in JIRA.".format(self.link(subtask_id))
            self._createPlanRecord(subtask_id)
            return subtask_id
        else:
            error("Failed to create subtask in JIRA.", response)

    def updateIssue(self, issue_id, new_status, comment, new_summary = None, add_record = True):
        # Check for ticket
        self._checkIssue(issue_id)

        # Status transition
        if new_status:
            self._transitionIssue(issue_id, new_status, new_summary)
            if comment:
                self._updateComment(issue_id, comment)
        else:
            self._updateIssue(issue_id, new_summary, comment)

        print u"Updated {0}".format(self.link(issue_id))
        if add_record:
            self._createPlanRecord(issue_id)

    def checkQuery(self, query):
        if not query:
            error('No JQL query provided.')

        # Collect raw data based on hosting type
        request_body = {
            'jql': query,
            'fields': ['summary', 'status']
        }

        if self._getHostingOption() == 'CLOUD':
            issue_data = self._executeCloudSearch(request_body, max_results=None)
        else:
            issue_data = self._executeServerSearch(request_body, max_results=None)

        # Process the collected data
        issues = {}
        for item in issue_data:
            issue = item['key']
            issues[issue] = (item['fields']['summary'], item['fields']['status']['name'])

        return issues

    def createVersion(self, projectId, versionName, archived, description, releaseDate, released):
        versionEndpoint = self._getVersionUri() + "/version"
        if not projectId:
            error("No project id provided")
        request = self._createRequest()
        formattedDate = None
        if releaseDate:
            format_out = SimpleDateFormat("yyyy-MM-dd")
            formattedDate = str(format_out.format(releaseDate))
        payload = {
            'archived': archived,
            'description': description,
            'name': versionName,
            'projectId': projectId,
            'releaseDate': formattedDate,
            'released': released
        }
        response = request.post(versionEndpoint, self._serialize(payload), contentType="application/json", headers=self._createApiTokenHeader())
        if response.status == 201:
            data = Json.loads(response.response)
            version_id = data["id"]
            if version_id:
                print("Version is created for the project : '{0}'".format(versionName))
                return version_id
            else:
                error("Failed to retrieve Version ID.")
        else:
            error("Failed to create Version : ", response)

    def getVersionIdsForProject(self, projectId):
        if not projectId:
            error("No project id provided.")
        request = self._createRequest()
        response = request.get(self._versionsUrl(projectId), contentType="application/json",
                                headers=self._createApiTokenHeader())
        if response.status != 200:
            error(u"Unable to find versions for project id %s" % projectId, response)
        versionIds = []
        for item in Json.loads(response.response):
            versionIds.append(item['id'])
        if not versionIds:
            print "No versions found for project id %s" % projectId
        else:
            print str(versionIds) + "\n"
        return versionIds

    def queryForFields(self, query, fields):
        if not query:
            error('No JQL query provided.')

        # Collect issue data based on hosting type
        request_body = {
            'jql': query,
            'fields': fields
        }

        if self._getHostingOption() == 'CLOUD':
            issue_data = self._executeCloudSearch(request_body, 1000)
        else:
            issue_data = self._executeServerSearch(request_body)

        # Process the collected data
        issues = []
        for item in issue_data:
            issue = {}
            for field in fields:
                issue[field] = item[field]
            issues.append(issue)

        return issues

    def queryForIssueIds(self, query):
        issues = self.queryForFields(query, ["id"])
        # for backwards compatibility, return a flat list with ids
        return [x["id"] for x in issues]

    def get_boards(self, board_name):
        if not board_name:
            error("No board name provided.")
        request = self._createRequest()
        response = request.get("/rest/agile/1.0/board?name=%s" % board_name, contentType="application/json",
                               headers=self._createApiTokenHeader())
        if response.status != 200:
            error(u"Unable to find boards for {0}".format(board_name), response)
        return Json.loads(response.response)['values']

    def get_all_sprints(self, board):
        if not board:
            error("No board id provided")
        request = self._createRequest()
        response = request.get("/rest/agile/1.0/board/%s/sprint" % board["id"], contentType="application/json",
                               headers=self._createApiTokenHeader())
        sprints = {}
        if response.status != 200:
            error(u"Unable to find sprints for board {0}".format(board["name"]), response)
        sprints_json = Json.loads(response.response)['values']
        for sprint_json in sprints_json:
            sprints[sprint_json["name"]] = sprint_json["id"]
            print "| %s | %s | %s | %s |" % (sprint_json["name"], sprint_json["id"],
                                             sprint_json["startDate"] if sprint_json.has_key(
                                                 "startDate") else "not defined",
                                             sprint_json["endDate"] if sprint_json.has_key(
                                                 "endDate") else "not defined")
        return sprints

    def updateLabels(self, issue_id, labels):
        # Check for ticket
        self._checkIssue(issue_id)

        updates = []
        for key in labels.keys():
            updates.append({labels[key]: key})
        body = {'update': {'labels': updates}}
        request = self._createRequest()
        response = request.put(self._issueUrl(issue_id), self._serialize(body), contentType='application/json', headers=self._createApiTokenHeader())
        if response.status != 204:
            error(u"Unable to update issue {0}. Please make sure the issue is not in a 'closed' state.".format(self.link(issue_id)), response)

    def updateFixVersions(self, issue_id, fixVersions):
        # Check for ticket
        self._checkIssue(issue_id)

        updates = []
        for key in fixVersions.keys():
            updates.append({fixVersions[key]: {'name': key}})
        body = {'update': {'fixVersions': updates}}
        request = self._createRequest()
        response = request.put(self._issueUrl(issue_id), self._serialize(body), contentType='application/json', headers=self._createApiTokenHeader())
        if response.status != 204:
            error(u"Unable to update issue {0}. Please make sure the issue is not in a 'closed' state.".format(self.link(issue_id)), response)

    def _getUpdatedIssueData(self, summary, comment):
        updated_data = {}

        if comment:
            updated_data.update({
                "comment": [
                    {
                        "add": {
                            "body": comment
                        }
                    }
                ]
            })

        if summary is not None:
            updated_data.update({
                "summary": [
                    {
                        "set": summary
                    }
                ]
            })

        return updated_data

    def _updateComment(self, issue_id, comment):
        request = self._createRequest()
        response = request.post(self._issueUrl(issue_id) + "/comment", self._serialize({"body": comment}),
                                contentType='application/json', headers=self._createApiTokenHeader())

        if response.status != 201:
            error(u"Unable to comment issue {0}. Make sure you have 'Add Comments' permission.".format(self.link(issue_id)), response)

    def _updateIssue(self, issue_id, new_summary, comment):

        updated_data = self._getUpdatedIssueData(new_summary, comment)

        # Create POST body
        request_data = {"update": updated_data}

        # Do request
        request = self._createRequest()
        response = request.put(self._issueUrl(issue_id), self._serialize(request_data),
                               contentType='application/json', headers=self._createApiTokenHeader())

        # Parse result
        if response.status != 204:
            error(u"Unable to update issue {0}. Please make sure the issue is not in a 'closed' state.".format(self.link(issue_id)), response)

    def _transitionIssue(self, issue_id, new_status, summary):

        issue_url = self._issueUrl(issue_id)

        # Find possible transitions
        request = self._createRequest()
        response = request.get(issue_url + "/transitions?expand=transitions.fields",
                               contentType='application/json', headers=self._createApiTokenHeader())

        if response.status != 200:
            error(u"Unable to find transitions for issue {0}".format(self.link(issue_id)), response)

        transitions = Json.loads(response.response)['transitions']

        # Check  transition
        wanted_transaction = -1
        for transition in transitions:
            if transition['to']['name'].lower() == new_status.lower():
                wanted_transaction = transition['id']
                break

        if wanted_transaction == -1:
            error(u"Unable to find status {0} for issue {1}".format(new_status, self.link(issue_id)))

        # Prepare POST body
        transition_data = {
            "transition": {
                "id": wanted_transaction
            }
        }

        # Perform transition
        response = request.post(issue_url + "/transitions?expand=transitions.fields", self._serialize(transition_data),
                                contentType='application/json', headers=self._createApiTokenHeader())

        if response.status != 204:
            error(u"Unable to perform transition {0} for issue {1}".format(wanted_transaction, self.link(issue_id)), response)

        if summary is not None:
            updated_data = {
                "summary": [
                    {
                        "set": summary
                    }
                ]
            }
            # Create POST body
            request_data = {"update": updated_data}

            # Do request
            request = self._createRequest()
            response = request.put(self._issueUrl(issue_id), self._serialize(request_data),
                                   contentType='application/json', headers=self._createApiTokenHeader())
            # Parse result
            if response.status != 204:
                error(u"Unable to update issue {0}. Please make sure the issue is not in a 'closed' state.".format(self.link(issue_id)), response)

    def _createRequest(self):
        params = self.jira_server.copy()

        if params['apiToken'] or ('pat' in params and params['pat']):
            params.pop('username')
            params.pop('password')
            if params['apiToken']:
                if 'pat' in params:
                    params.pop('pat')
            elif params['pat']:
                if 'apiToken' in params:
                    params.pop('apiToken')
        if self.username and self.password and not self.api_token and not self.pat:
            params['username'] = self.username
            params['password'] = self.password

        return HttpRequest(params)

    def _createApiTokenHeader(self):
        headers = None
        if self.pat:
            headers = {'Authorization': 'Bearer %s' % self.pat}
        elif self.jira_server['pat']:
            headers = {'Authorization': 'Bearer %s' % self.jira_server['pat']}
        elif self.api_token and self.username:
            headers = {'Authorization': 'Basic %s' % b64encode('%s:%s' % (self.username, self.api_token))}
        elif self.password and self.username:
            headers = None
        elif self.jira_server['apiToken']:
            headers = {'Authorization': 'Basic %s' % b64encode('%s:%s' % (self.jira_server['username'], self.jira_server['apiToken']))}

        return headers

    def link(self, issue_id):
        return "[{0}]({1}/browse/{0})".format(issue_id, self.jira_server['url'].rstrip('/'))

    def _issueUrl(self, issue_id):
        return self._getVersionUri() + "/issue/" + issue_id

    def _checkIssue(self, issue_id):
        request = self._createRequest()
        response = request.get(self._issueUrl(issue_id),
                               contentType='application/json', headers=self._createApiTokenHeader())

        if response.status != 200:
            error(u"Unable to find issue {0}".format(self.link(issue_id)), response)

    def _versionsUrl(self, projectId):
        return self._getVersionUri() + "/project/%s/versions" % projectId

    def _serialize(self, content):
        return Json.dumps(content)

    def _getHostingOption(self):
        if self.pat:
            hosting_option = 'SERVER'
        elif self.api_token and self.username:
            hosting_option = 'CLOUD'
        elif self.password and self.username:
            hosting_option = 'SERVER'
        elif self.jira_server['apiToken']:
            hosting_option = 'CLOUD'
        else:
            hosting_option = 'SERVER'

        return hosting_option

    def _getVersionUri(self):
        return '/rest/api/2'

    def _getSearchEndpoint(self):
        """
        Returns the appropriate search endpoint based on JIRA hosting type.
        JIRA Cloud: Uses enhanced /search/jql endpoint
        JIRA Server: Uses traditional /search endpoint
        """
        if self._getHostingOption() == 'CLOUD':
            return '/search/jql'
        else:
            return '/search'

    def _createPlanRecord(self, issue_id):
        if self.task_reporting_api and self.task:
            try:
                self.query("key=" + issue_id, quiet = True)
            except:
                print "\n\n"
                print sys.exc_info()

    def _executeCloudSearch(self, request_body, max_results=MAX_RESULTS):
        """
        Utility method to execute paginated search requests for JIRA Cloud
        Args:
            request_body: The base request payload
            max_results: Maximum number of results to return. Defaults to MAX_RESULTS (1000).
                        If set to None, returns all available results without limit.
            endpoint_suffix: The API endpoint suffix to use
        """
        page_token = None
        all_issues = []

        while True:
            # Add pagination token if available
            current_request = request_body.copy()
            if page_token:
                current_request['nextPageToken'] = page_token

            # Execute request
            request = self._createRequest()
            endpoint = self._getVersionUri() + "/search/jql"
            response = request.post(endpoint, self._serialize(current_request),
                                    contentType='application/json', headers=self._createApiTokenHeader())

            if response.status == 200:
                response_data = Json.loads(response.response)
                if not response_data.get('issues'):
                    break
   
                page_issues = response_data['issues']

                if max_results is not None:
                    #  Fetch limited records
                    remaining_slots = max_results - len(all_issues)
                    if len(page_issues) <= remaining_slots:
                        all_issues.extend(page_issues)
                    else:
                        all_issues.extend(page_issues[:remaining_slots])
                        break
 
                    if len(all_issues) >= max_results:
                        break
                else:
                    # Fetch all records
                    all_issues.extend(page_issues)

                # Check pagination status
                if response_data.get('isLast', True):
                    break
                page_token = response_data.get('nextPageToken')
                if not page_token:
                    break
            else:
                error(u"Failed to execute search in JIRA.", response)

        return all_issues

    def _executeServerSearch(self, request_body, max_results=MAX_RESULTS):
        """
        Utility method to execute paginated search requests for JIRA Server
        Args:
            request_body: The base request payload
            max_results: Maximum number of results to return. Defaults to MAX_RESULTS (1000).
                    If set to None, returns all available results without limit.
            endpoint_suffix: The API endpoint suffix to use
        """
        max_per_page = 1000
        current_offset = 0
        all_issues = []

        while True:
            # Update request with pagination
            current_request = request_body.copy()
            current_request['startAt'] = current_offset
            current_request['maxResults'] = max_per_page

            # Execute request
            request = self._createRequest()
            endpoint = self._getVersionUri() + "/search"
            response = request.post(endpoint, self._serialize(current_request),
                                contentType='application/json', headers=self._createApiTokenHeader())

            if response.status == 200:
                response_data = Json.loads(response.response)
                if not response_data.get('issues'):
                    break

                page_issues = response_data['issues']

                if max_results is not None:
                    # Fetch limited records
                    remaining_slots = max_results - len(all_issues)
                    if len(page_issues) <= remaining_slots:
                        all_issues.extend(page_issues)
                    else:
                        all_issues.extend(page_issues[:remaining_slots])
                        break

                    if len(all_issues) >= max_results:
                        break
                else:
                    # Fetch all records 
                    all_issues.extend(page_issues)

                current_offset += len(page_issues)

                # Check if we've reached the end
                if len(page_issues) < max_per_page:
                    break
            else:
                error(u"Failed to execute search in JIRA.", response)

        return all_issues
