package com.xebialabs.xlrelease.plugins.jenkins;

import com.nimbusds.jose.*;
import com.nimbusds.jose.crypto.*;
import com.nimbusds.jwt.*;

import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import java.util.Date;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.RSAPrivateKey;

/**
 * JwtSigner is a utility class for creating and signing JSON Web Tokens (JWTs).
 * It supports multiple signing algorithms such as RSA and ECDSA.
 */
public class JwtSigner {

    /**
     * Signs a JWT using the specified algorithm and private key.
     *
     * @param algorithm       The signing algorithm (e.g., RS256, ES256).
     * @param privateKeyString The private key in PEM format as a string.
     * @param privateKeyId    The key ID to include in the JWT header (optional).
     * @param audience        The audience claim for the JWT.
     * @param issuer          The issuer claim for the JWT.
     * @param subject         The subject claim for the JWT.
     * @return The serialized JWT as a string.
     * @throws Exception If an error occurs during signing or key conversion.
     */
    public static String signJwt(String algorithm, String privateKeyString, String privateKeyId, String audience,
                                 String issuer, String subject) throws Exception {

        // Determine JWS algorithm
        JWSAlgorithm alg = JWSAlgorithm.parse(algorithm);

        // Convert string private key to PrivateKey type based on algorithm
        String keyAlgorithm = alg.getName().startsWith("RS") || alg.getName().startsWith("PS") ? "RSA" : "EC";
        PrivateKey privateKey = convertStringToPrivateKey(privateKeyString, keyAlgorithm);

        // Create signer based on algorithm type
        JWSSigner signer;
        if (alg.getName().startsWith("RS") || alg.getName().startsWith("PS")) {
            signer = new RSASSASigner((RSAPrivateKey) privateKey);
        } else if (alg.getName().startsWith("ES")) {
            signer = new ECDSASigner((ECPrivateKey) privateKey);
        } else {
            throw new IllegalArgumentException("Unsupported algorithm: " + alg);
        }

        // Build JWT claims
        JWTClaimsSet.Builder claimsBuilder = new JWTClaimsSet.Builder()
            .issuer(issuer)
            .subject(subject)
            .audience(audience)
            .issueTime(new Date())
            .expirationTime(new Date(System.currentTimeMillis() + 10 * 60 * 1000))
            .claim("target_audience", audience);

        JWTClaimsSet claims = claimsBuilder.build();
        
        JWSHeader.Builder headerBuilder = new JWSHeader.Builder(alg).type(JOSEObjectType.JWT);
        if (privateKeyId != null) {
            headerBuilder.keyID(privateKeyId); 
        }
        // Create and sign the JWT
        SignedJWT jwt = new SignedJWT(headerBuilder.build(), claims);
        jwt.sign(signer);

        return jwt.serialize();
    }

    /**
     * Converts a private key string in PEM format to a PrivateKey object.
     *
     * @param privateKeyString The private key in PEM format as a string.
     * @param algorithm        The algorithm of the private key (e.g., RSA, EC).
     * @return The PrivateKey object.
     * @throws Exception If an error occurs during key conversion.
     */
    public static PrivateKey convertStringToPrivateKey(String privateKeyString, String algorithm) throws Exception {
        if (privateKeyString == null || privateKeyString.trim().isEmpty()) {
            throw new IllegalArgumentException("Private key string cannot be null or empty.");
        }

        // Normalize PEM format and remove headers/footers
        privateKeyString = privateKeyString
            .replace("\\n", "\n")
            .replace("\r\n", "\n")
            .replace("\r", "\n")
            .replace("-----BEGIN PRIVATE KEY-----", "")
            .replace("-----END PRIVATE KEY-----", "")
            .replaceAll("\\s", "");

        byte[] keyBytes;
        try {
            keyBytes = Base64.getDecoder().decode(privateKeyString);
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Invalid Base64 private key format.", e);
        }

        PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(algorithm);

        return keyFactory.generatePrivate(spec);
    }
}