package com.xebialabs.xlrelease.webhooks.github

import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlplatform.webhooks.authentication.RequestAuthenticationMethod
import com.xebialabs.xlplatform.webhooks.domain.Endpoint
import grizzled.slf4j.Logger
import org.apache.commons.codec.binary.Hex

import java.nio.charset.StandardCharsets
import java.security.MessageDigest
import javax.crypto.Mac
import javax.crypto.spec.SecretKeySpec
import scala.util.{Failure, Try}

object GithubAuthenticationMethod extends RequestAuthenticationMethod {
  val SIGNATURE_HEADER = "X-Hub-Signature"
  val SIGNATURE_PREFIX = "sha1="
  val HMAC_SHA1_ALGORITHM = "HmacSHA1"

  // remove ugly '$' at the end of this singleton when logging
  lazy val logger: Logger = Logger(this.getClass.getName.dropRight(1))

  lazy val passwordEncrypter: PasswordEncrypter = PasswordEncrypter.getInstance()

  override def authenticateScala(endpoint: Endpoint,
                                 headers: Map[String, String],
                                 params: Map[String, Array[String]],
                                 payload: String): Boolean = {
    val authentication: Try[Boolean] = for {
      signature <- getSignature(headers)
      key <- getKey(endpoint.authentication.asInstanceOf[GithubAuthentication])
      computed <- computeSignature(payload, key)
    } yield {
      logger.trace(s"computed: ${Hex.encodeHex(computed).mkString("")}")
      MessageDigest.isEqual(computed, signature)
    }

    authentication.recover {
      case err =>
        Option(err.getMessage).foreach(logger.warn(_))
        false
    }.get
  }

  def getSignature(headers: Map[String, String]): Try[Array[Byte]] = {
    headers.get(SIGNATURE_HEADER).collect {
      case signature if signature startsWith SIGNATURE_PREFIX =>
        logger.trace(s"Got signature: $signature")
        Try(Hex.decodeHex(signature.drop(SIGNATURE_PREFIX.length)))
    }.getOrElse(Failure(new IllegalArgumentException(s"Header '$SIGNATURE_HEADER' not found.")))
  }

  def getKey(requestAuthenticationConfig: GithubAuthentication): Try[SecretKeySpec] = {
    Option(requestAuthenticationConfig.githubSecret).collect {
      case githubSecret if githubSecret.nonEmpty =>
        logger.trace(s"githubSecret: $githubSecret")
        Try(new SecretKeySpec(passwordEncrypter.decrypt(githubSecret).getBytes, HMAC_SHA1_ALGORITHM))
    }.getOrElse {
      Failure(new IllegalStateException(s"'$requestAuthenticationConfig' has no githubSecret configured!"))
    }
  }

  def computeSignature(payload: String, key: SecretKeySpec): Try[Array[Byte]] = Try {
    val hmac: Mac = Mac.getInstance(HMAC_SHA1_ALGORITHM)
    hmac.init(key)
    hmac.doFinal(payload.replace("\r\n", "\n").getBytes(StandardCharsets.UTF_8))
  }

}