#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
#
import re
import time
from datetime import datetime

from fortify import FortifyService
from fortify.concurrency.FortifyComplianceTileTasks import GetVersionVulnerabilitiesTask
from fortify.concurrency.FortifyConcurrentExecutorService import FortifyConcurrentExecutorService

global fortifyServer, projectName, tile, standards, timeFrame, dateFrom, dateTo, versionFilterRegex


def version_filter(version_regex, from_date, to_date):
    def do_filter(version_name, date):
        date = time.mktime(datetime.strptime(date[:19], '%Y-%m-%dT%H:%M:%S').timetuple())
        return from_date <= date <= to_date and re.search(version_regex, version_name, flags=re.IGNORECASE) is not None

    return do_filter


# Validate properties
missedProperties = []
if not fortifyServer:
    missedProperties.append('Fortify server')
if not projectName:
    missedProperties.append('Application name')
if not standards:
    missedProperties.append('Standards')
if not timeFrame:
    missedProperties.append('Time frame')

if missedProperties:
    raise Exception("{} must be provided".format(', '.join(missedProperties)))

# Fetch max 30 versions
LIMIT = 200
MAX_VERSIONS = tile.getProperty('maxResults')
version_filter_func = version_filter(versionFilterRegex, timeFrame.getStartDate(dateFrom) / 1000.0, timeFrame.getEndDate(dateTo) / 1000.0)

executor = FortifyConcurrentExecutorService()
fields = ['id', 'name', 'creationDate', 'currentState']
versions = []
offset = 0
while len(versions) < MAX_VERSIONS:
    project_versions = FortifyService.get_project_versions(fortifyServer, projectName, start=offset, limit=LIMIT, fields=fields)
    for version in project_versions.get('data'):
        if version_filter_func(version.get('name'), version.get('creationDate')) and len(versions) < MAX_VERSIONS:
            project_data = {
                'versionId': version.get('id'),
                'versionName': version.get('name'),
                'versionUrl': '{}/html/ssc/version/{}'.format(fortifyServer['url'].strip('/'), version.get('id')),
                'vulnerabilities': []
            }
            versions.append(project_data)
            hasAnalysisResults = version.get('currentState', {}).get('analysisResultsExist', True)
            if hasAnalysisResults:
                executor.add(GetVersionVulnerabilitiesTask(fortifyServer, version.get('id'), standards))

    count = project_versions.get('count')
    offset += LIMIT
    if offset >= count:
        break

# Fetch vulnerabilities in parallel
futures = executor.execute()
vulnerabilities = {}
for future in futures:
    result = future.get()
    if result.exception is not None:
        raise result.exception
    vulnerabilities[result.version_id] = result.result

# Set vulnerabilities to the corresponding project version
for version in versions:
    version['vulnerabilities'] = vulnerabilities.get(version.get('versionId'), [])

data = []
if versions:
    data = {
        'count': len(versions),
        'hasMore': len(versions) >= MAX_VERSIONS,
        'versions': versions
    }
