#
# Copyright (c) 2019. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
#

import json
import urllib

from xlrelease.HttpRequest import HttpRequest

API_BASE = "api/v1"


class FortifyService(object):

    @staticmethod
    def get_fortify_token_header(fortify_server):
        headers_fortify_token = {}
        params = fortify_server.copy()
        if params['password']:
            headers_fortify_token = {
                "Authorization": "FortifyToken {}".format(params['password'])
            }
        else:
            print("Token Not given, Please enter the Token.")
        return headers_fortify_token


    @staticmethod
    def get_fortify_performance_indicator_histories(fortify_server, version_id, indicator=''):
        header = FortifyService.get_fortify_token_header(fortify_server)
        http_request = FortifyService.get_http_request(fortify_server,header)
        fortify_response = http_request.get('{}/projectVersions/{}/performanceIndicatorHistories/{}'
                                            .format(API_BASE, version_id, indicator), headers=header)
        error_message = "Failed to get Fortify Measures. Server return [%s], with content [%s]" % (
            fortify_response.status, fortify_response.response)
        FortifyService.validate_response(error_message, fortify_response)
        return json.loads(fortify_response.getResponse())['data']

    @staticmethod
    def get_performance_indicators(params):
        header = FortifyService.get_fortify_token_header(params)
        http_request = FortifyService.get_http_request(params,header)
        fortify_response = http_request.get('{}/performanceIndicators'.format(API_BASE), headers=header)
        error_message = "Failed to get Fortify performance indicators. Server returns [%s], with content [%s]" % (
            fortify_response.status, fortify_response.response)
        FortifyService.validate_response(error_message, fortify_response)
        return json.loads(fortify_response.getResponse())['data']

    @staticmethod
    def get_issues_count(fortify_server, version_id, issue_type=None):
        header = FortifyService.get_fortify_token_header(fortify_server)
        http_request = FortifyService.get_http_request(fortify_server,header)
        encoded_params = urllib.quote("[fortify priority order]:{}".format(issue_type)) if issue_type else ''
        fortify_response = http_request.get('{}/projectVersions/{}/issues?limit=1&fields=friority&q={}&qm=issues'.format(API_BASE, version_id, encoded_params), headers=header)
        error_message = "Failed to get Fortify Measures. Server return [%s], with content [%s]" % (
            fortify_response.status, fortify_response.response)
        FortifyService.validate_response(error_message, fortify_response)
        return json.loads(fortify_response.getResponse())['count']

    @staticmethod
    def get_standards(fortify_server):
        all_versions = FortifyService.do_get(fortify_server, "{}/projectVersions?fields=id&limit=1".format(API_BASE))
        for version in all_versions:
            data = FortifyService.do_get(fortify_server, "{}/projectVersions/{}/issueSelectorSet?fields=groupBySet".format(API_BASE, version['id']))
            return filter(lambda x: x['entityType'] == 'EXTERNALLIST', data['groupBySet'])
        raise Exception("Unable to fetch standards due to missing project versions.")

    @staticmethod
    def do_get(fortify_server, url):
        header = FortifyService.get_fortify_token_header(fortify_server)
        http_request = FortifyService.get_http_request(fortify_server,header)
        fortify_response = http_request.get(url, headers=header)
        FortifyService.validate_response("Failed to GET %s. Server returned [%s], with content [%s]" % (
            url, fortify_response.status, fortify_response.response), fortify_response)
        return json.loads(fortify_response.getResponse())['data']

    @staticmethod
    def get_project_versions(fortify_server, project_name, project_version=None, start=0, limit=200, fields=None):
        header = FortifyService.get_fortify_token_header(fortify_server)
        http_request = FortifyService.get_http_request(fortify_server,header)
        search_query = 'project.name:"{}"'.format(project_name)
        if project_version is not None:
            search_query += '+name:"{}"'.format(project_version)
        if fields is None:
            fields = ['id', 'name', 'creationDate']
        fortify_response = http_request.get(
            '{}/projectVersions?fields={}&q={}&start={}&limit={}'.format(API_BASE, ','.join(fields), search_query, start, limit), headers=header)
        FortifyService.validate_response("Failed to load Fortify project version. Server return [%s], with content [%s]" %
                                         (fortify_response.status, fortify_response.response), fortify_response)
        return json.loads(fortify_response.getResponse())

    @staticmethod
    def get_vulnerabilities(fortify_server, project_version, groups):
        header = FortifyService.get_fortify_token_header(fortify_server)
        http_request = FortifyService.get_http_request(fortify_server,header)
        request_body = {
            "requests": []
        }
        for uuid_name in groups:
            security_uuid, security_name = uuid_name.split(':')

            request_body['requests'].append(
                {
                    "httpVerb": "GET",
                    "postData": {},
                    "uri": "{}/api/v1/projectVersions/{}/issueGroups?groupingtype={}&fields=cleanName,totalCount".format(fortify_server['url'].strip('/'),
                                                                                                                         project_version, security_uuid)
                }
            )
        url = '{}/bulk'.format(API_BASE)
        fortify_response = http_request.post(url, body=json.dumps(request_body), contentType='application/json', headers=header)
        return json.loads(fortify_response.getResponse())

    @staticmethod
    def validate_response(error_message, fortify_response):
        if not fortify_response.isSuccessful():
            raise Exception(error_message)

    @staticmethod
    def add_comment(comment):
        print "```"
        print comment
        print "```"

    @staticmethod
    def get_http_request(fortify_server,header):
        params = fortify_server.copy()
        if header is not None:
            params.pop('username')
            params.pop('password')
        http_request = HttpRequest(params)
        return http_request