#
# Copyright (c) 2019. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
#

import sys
import yaml
from datetime import datetime

from fortifyOnDemand import FortifyOnDemandService
from fortifyOnDemand.task import add_comment

global fortifyOnDemandServer, projectName, projectVersion, minimumStarRating, facetApi, task


def add_code_compliance_facet(outcome=None, compliance_data=None, analysis_date=None, project_url=None):
    try:
        facet = facetApi.newFacet("udm.CodeComplianceFacet")
        facet.targetId = task.id
        facet.serverUrl = fortifyOnDemandServer['url']
        facet.serverUser = fortifyOnDemandServer['username']
        facet.project = projectName
        facet.outcome = outcome
        facet.complianceData = yaml.dump(compliance_data, default_flow_style=False)
        facet.analysisDate = analysis_date
        facet.project_url = project_url
        facetApi.createFacet(facet)
    except:
        exctype, value = sys.exc_info()[:2]
        print("{} occurred while creating `udm.CodeComplianceFacet` - {}".format(exctype, value))


def convert_to_datetime(from_date):
    return datetime.strptime(from_date, '%Y-%m-%dT%H:%M:%S.%f')


if not fortifyOnDemandServer:
    raise Exception("Fortify on Demand server ID must be provided")

application_data = FortifyOnDemandService.get_application_data(fortifyOnDemandServer, application_name=projectName)

total_count = application_data["totalCount"]
items = application_data["items"]

if total_count > 0 and items[0]["applicationName"] == projectName:
    release_rating_data = FortifyOnDemandService.get_release_ratings(fortifyOnDemandServer,
                                                                     application_id=items[0]["applicationId"],
                                                                     release_name=projectVersion)
    release_total_count = release_rating_data["totalCount"]
    release_items = release_rating_data["items"]

    if release_total_count > 0 and release_items[0]["releaseName"] == projectVersion:
        analysis_date_list = []
        if release_items[0]["staticScanDate"] is not None:
            analysis_date_list.append(convert_to_datetime(release_items[0]["staticScanDate"]))
            add_comment("Static Analysis Date and Time : {}".format(release_items[0]["staticScanDate"]))

        if release_items[0]["dynamicScanDate"] is not None:
            analysis_date_list.append(convert_to_datetime(release_items[0]["dynamicScanDate"]))
            add_comment("Dynamic Analysis Date and Time : {}".format(release_items[0]["dynamicScanDate"]))

        if release_items[0]["mobileScanDate"] is not None:
            analysis_date_list.append(convert_to_datetime(release_items[0]["mobileScanDate"]))
            add_comment("Mobile Analysis Date and Time : {}".format(release_items[0]["mobileScanDate"]))

        last_analysis_date = None if not analysis_date_list else max(analysis_date_list)
        scan_failed = release_items[0]["rating"] < minimumStarRating
        project_url = "{}/Releases/{}/Overview".format(fortifyOnDemandServer['url'].replace('api.', '').strip('/'),
                                                       release_items[0]["releaseId"])

        metrics = {"rating": "Fortify Security Rating", "issueCount": "Total Issues", "critical": "Critical Issues",
                   "high": "High Issues", "medium": "Medium Issues", "low": "Low Issues"}
        data = {}
        for key, value in metrics.iteritems():
            data[str(value)] = release_items[0][key] if key in release_items[0] else None

        add_code_compliance_facet(analysis_date=last_analysis_date,
                                  outcome=("FAILED" if scan_failed else "PASSED"),
                                  compliance_data=data,
                                  project_url=project_url)

        add_comment("Fortify Security Rating : {}".format(release_items[0]["rating"]))
        add_comment("Minimum Security Rating: {}".format(minimumStarRating))

        if scan_failed:
            raise Exception("Code compliance check(s) failed")
        add_comment("Code compliance checks verified.")
    else:
        add_code_compliance_facet()
        error_message = "Failed to get release data. Please verify Release Name : %s on server." % projectVersion
        raise Exception(error_message)
else:
    add_code_compliance_facet()
    error_message = "Failed to get application data. Please verify Application Name : %s on server." % projectName
    raise Exception(error_message)
