package com.xebialabs.xlrelease.runner.docker

import com.github.dockerjava.api.DockerClient
import com.github.dockerjava.core.{DefaultDockerClientConfig, DockerClientConfig, DockerClientImpl}
import com.github.dockerjava.httpclient5.ApacheDockerHttpClient
import com.github.dockerjava.transport.DockerHttpClient
import com.xebialabs.xlrelease.runner.docker.domain.DockerOptions
import com.xebialabs.xlrelease.runner.domain.RunnerId

import java.util.concurrent.ConcurrentHashMap

object DockerClientFactory {

  private val clientLookupMap: ConcurrentHashMap[RunnerId, (DockerOptions, DockerClient)] = new ConcurrentHashMap[RunnerId, (DockerOptions, DockerClient)]()

  def apply(dockerOptions: DockerOptions): DockerClient = {
    clientLookupMap.computeIfAbsent(dockerOptions.runnerId, _ => (dockerOptions, createClient(dockerOptions))) match {
      case (dOptions, dClient) =>
        if (dOptions == dockerOptions) {
          dClient
        } else {
          clientLookupMap.synchronized {
            val updatedClient = createClient(dockerOptions)
            clientLookupMap.replace(dockerOptions.runnerId, (dockerOptions, updatedClient))
            updatedClient
          }
        }
    }
  }

  private def createClient(dockerOptions: DockerOptions): DockerClient = {
    val registryOptions = dockerOptions.registryOptions
    val config: DockerClientConfig = DefaultDockerClientConfig.createDefaultConfigBuilder()
      .withDockerHost(dockerOptions.host) // or detect default
      .withDockerTlsVerify(false)
      //.withDockerCertPath("/home/user/.docker")
      //.withRegistryUsername("digitalai")
      //.withRegistryPassword("<your_encrypted_password>")
      //.withRegistryEmail(registryMail)
      .withRegistryUrl(registryOptions.registryUrl)
      .build()

    val httpClient: DockerHttpClient = new ApacheDockerHttpClient.Builder()
      .dockerHost(config.getDockerHost)
      .sslConfig(config.getSSLConfig)
      .maxConnections(dockerOptions.capacity * 2) //TODO: check if this is good enough
      //.connectionTimeout(Duration.ofSeconds(30))
      //.responseTimeout(Duration.ofSeconds(45))
      .build()

    DockerClientImpl.getInstance(config, httpClient)
  }
}
