package com.xebialabs.xlrelease.runner.docker.actors

import akka.actor.{Actor, ActorLogging, ActorRef}
import com.github.dockerjava.api.DockerClient
import com.github.dockerjava.api.async.ResultCallback
import com.github.dockerjava.api.command.{InspectContainerResponse, WaitContainerResultCallback}
import com.github.dockerjava.api.model.{Container, Frame, PullResponseItem, WaitResponse}
import com.github.dockerjava.core.util.CompressArchiveUtil
import com.xebialabs.xlrelease.Environment
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.runner.docker.actors.DockerJobExecutorActor._
import com.xebialabs.xlrelease.runner.domain.JobId
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration.URI_SCHEME_LOCAL_STORAGE
import com.xebialabs.xlrelease.storage.domain.LogEntry
import com.xebialabs.xlrelease.storage.service.StorageService
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream
import org.apache.commons.io.FileUtils
import org.apache.commons.io.output.ByteArrayOutputStream
import org.apache.hc.core5.http.ConnectionClosedException
import org.apache.http.{ConnectionClosedException => HttpClient4ConnectionClosedException}
import org.springframework.util.StreamUtils
import org.springframework.util.StringUtils.hasText

import java.io.BufferedOutputStream
import java.net.{ConnectException, SocketException}
import java.nio.charset.StandardCharsets
import java.nio.file.Files
import java.time.Instant
import java.time.format.DateTimeFormatter
import scala.concurrent.duration.DurationLong
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try, Using}

trait DockerService {
  it: Actor with ActorLogging =>
  def selfRef: ActorRef

  def xlrConfig: XlrConfig

  def storageService: StorageService

  implicit def dockerClient: DockerClient

  //noinspection ScalaStyle
  def pullImage(taskId: String, jobId: JobId, taskImg: String, lastChunk: Long): Unit = {
    try {
      dockerClient.pullImageCmd(taskImg).exec(new ResultCallback.Adapter[PullResponseItem] with CommandSupport {
        private val logHelper = new DockerLogHelper(taskId, jobId, lastChunk)
        private var closed: Boolean = false

        private def buildPayload(response: PullResponseItem): String = {
          val builder = new StringBuilder()
          if (null != response.getId) {
            builder.append(s"${response.getId}: ")
          }
          builder.append(s"${response.getStatus} ")
          if (null != response.getProgressDetail && null != response.getProgressDetail.getCurrent && null != response.getProgressDetail.getTotal) {
            val currentSize = FileUtils.byteCountToDisplaySize(response.getProgressDetail.getCurrent)
            val totalSize = FileUtils.byteCountToDisplaySize(response.getProgressDetail.getTotal)
            builder.append(s"[$currentSize/$totalSize]")
          }
          if (null != response.getErrorDetail) {
            builder.append(s"[${response.getErrorDetail.getCode}: ${response.getErrorDetail.getMessage}]")
          }
          builder.append(System.lineSeparator())
          builder.toString()
        }

        override def close(): Unit = {
          super.close()
          if (!closed) {
            addLogEntry(logHelper.flush())
          }
          closed = true
          logHelper.close()
        }

        override def onNext(response: PullResponseItem): Unit = {
          super.onNext(response)
          if (!closed) {
            val payload = buildPayload(response).getBytes()
            logHelper.appendEntry(payload).foreach(addLogEntry)
          }
        }

        override def onComplete(): Unit = {
          Try(super.onComplete()) match {
            case Failure(ex) =>
              handlePullImageFailure(ex)
            case Success(_) =>
              submitCommand(DockerOperationSuccess(jobId, PullImageCompleted))
          }
        }

        override def onError(throwable: Throwable): Unit = {
          Try(super.onError(throwable)) match {
            case Failure(ex) =>
              handlePullImageFailure(ex)
            case Success(_) =>
              throwable match {
                case e@(_: ConnectionClosedException | _: HttpClient4ConnectionClosedException | _: ConnectException | _: SocketException) =>
                  submitCommand(DockerOperationFailure(jobId, event = ConnectionFailed(e.getMessage)))
                case re: RuntimeException if re.getCause.isInstanceOf[ConnectException] =>
                  submitCommand(DockerOperationFailure(jobId, event = ConnectionFailed(re.getCause.getMessage)))
                case _ =>
                  handlePullImageFailure(throwable)
              }
          }
        }

        override def actorRef: ActorRef = selfRef

        private def handlePullImageFailure(ex: Throwable): Unit = {
          val errorMessage = s"Unable to pull image [$taskImg] for jobId [$jobId]"
          log.error(ex, errorMessage)
          submitCommand(DockerOperationFailure(jobId, OperationFailed(s"$errorMessage: ${ex.getMessage}")))
        }
      })
    } catch {
      case ex: Throwable =>
        val errorMessage = s"Unable to pull image [$taskImg] for jobId [$jobId]"
        log.error(ex, errorMessage)
        selfRef ! DockerOperationFailure(jobId, OperationFailed(s"$errorMessage: ${ex.getMessage}"))
    }
  }

  def createContainer(jobId: JobId, taskImg: String, containerName: String): Unit = {
    Try {
      val filteredContainers = getContainersByName(containerName)
      if (filteredContainers.isEmpty) {
        dockerClient.createContainerCmd(taskImg).withName(containerName).exec().getId
      } else {
        filteredContainers.head.getId
      }
    } match {
      case Failure(ex) =>
        val errorMessage = s"Unable to create a container for jobId [$jobId]"
        handleFailure(ex, jobId, errorMessage)
      case Success(containerId) =>
        selfRef ! DockerOperationSuccess(jobId, CreateContainerCompleted(containerId))
    }
  }

  def createContainerInputContext(jobId: JobId, containerId: String, inputContextJson: String): Unit = {
    Try {
      val tempDir = Files.createTempDirectory("container-task")
      val file = Files.createFile(tempDir.resolve("input"))
      FileUtils.writeStringToFile(file.toFile, inputContextJson, StandardCharsets.UTF_8)

      val temp = Files.createTempFile("", ".tar.gz")
      CompressArchiveUtil.tar(file, temp, true, false)

      Using(Files.newInputStream(temp)) { uploadStream =>
        dockerClient
          .copyArchiveToContainerCmd(containerId)
          .withTarInputStream(uploadStream)
          .exec()
      }
    } match {
      case Failure(ex) =>
        val errorMessage = s"Unable to create a container input context for jobId [$jobId] with containerId [$containerId]"
        handleFailure(ex, jobId, errorMessage)
      case Success(_) =>
        selfRef ! DockerOperationSuccess(jobId, CreateContainerInputContextCompleted)
    }
  }

  def startContainer(jobId: JobId, containerId: String): Unit = {
    Try {
      // start the container only if it is in created state
      if (getContainerState(containerId).getStatus.toLowerCase == "created") {
        dockerClient.startContainerCmd(containerId).exec()
      }
    } match {
      case Failure(ex) =>
        val errorMessage = s"Unable to start a container [$containerId] for jobId [$jobId]"
        handleFailure(ex, jobId, errorMessage)
      case Success(_) =>
        selfRef ! DockerOperationSuccess(jobId, StartContainerCompleted)
    }
  }

  def captureContainerLog(taskId: String, jobId: JobId, containerId: String, lastLogChunk: Long, lastLogTimestamp: String): Unit = {
    try {
      val timestamp = Instant.parse(lastLogTimestamp).getEpochSecond
      val cmd = dockerClient.logContainerCmd(containerId)
        .withStdOut(true)
        .withStdErr(true)
        .withFollowStream(true)
        .withTimestamps(true)
        .withSince(timestamp.toInt) // not so great API
      //.withUntil()
      cmd.exec(logCallback(taskId, jobId, lastLogChunk))
    } catch {
      case ex: Throwable =>
        val errorMessage = s"Unable to capture logs for jobId [$jobId] from containerId [$containerId]"
        log.error(ex, errorMessage)
        selfRef ! DockerOperationFailure(jobId, OperationFailed(s"$errorMessage: ${ex.getMessage}"))
    }
  }

  private def addLogEntry(logEntry: LogEntry): Unit = {
    val storedEntryUri = storageService.store(logEntry)
    log.debug(s"Sending AddLogEntry for log entry: $logEntry")
    selfRef ! AddLogEntry(logEntry, storedEntryUri)
  }

  private def logCallback(taskId: String, jobId: JobId, lastChunk: Long): ResultCallback[Frame] = new ResultCallback.Adapter[Frame] with CommandSupport {
    private val logHelper = new DockerLogHelper(taskId, jobId, lastChunk)
    private var closed: Boolean = false

    override def onNext(frame: Frame): Unit = {
      val payload = frame.getPayload
      if (!closed) {
        logHelper.appendEntry(payload).foreach(addLogEntry)
      } else {
        log.error("appending log entry AFTER close")
      }
    }

    override def close(): Unit = {
      super.close()
      if (!closed) {
        addLogEntry(logHelper.flush())
      } else {
        log.error("appending log entry AFTER close")
      }
      closed = true
      logHelper.close()
    }

    override def onComplete(): Unit = {
      Try(super.onComplete()) match {
        case Failure(ex) =>
          handleLogRetrievalFailure(ex)
        case Success(_) =>
          submitCommand(DockerOperationSuccess(jobId, event = LogRetrievalCompleted))
      }
    }

    override def onError(throwable: Throwable): Unit = {
      Try(super.onError(throwable)) match {
        case Failure(exception) =>
          handleLogRetrievalFailure(exception)
        case Success(_) =>
          // TODO handle this error a bit better - if it's connect exception it means we cannot even connect to Docker
          //  so do not just fail but switch to "FAILING" state until someone or something determines final state
          throwable match {
            case e@(_: ConnectionClosedException | _: HttpClient4ConnectionClosedException | _: ConnectException | _: SocketException) =>
              submitCommand(DockerOperationFailure(jobId, event = ConnectionFailed(e.getMessage)))
            case re: RuntimeException if re.getCause.isInstanceOf[ConnectException] =>
              submitCommand(DockerOperationFailure(jobId, event = ConnectionFailed(re.getCause.getMessage)))
            case _ =>
              handleLogRetrievalFailure(throwable)
          }
      }
    }

    override def actorRef: ActorRef = selfRef

    private def handleLogRetrievalFailure(ex: Throwable): Unit = {
      val errorMessage = s"Unable to get logs for jobId [$jobId]"
      log.error(ex, errorMessage)
      submitCommand(DockerOperationFailure(jobId, OperationFailed(s"$errorMessage: ${ex.getMessage}")))
    }
  }

  def waitForContainer(jobId: JobId, containerId: String): Unit = {
    try {
      dockerClient.waitContainerCmd(containerId).exec(new WaitContainerResultCallback() with CommandSupport {
        private var lastStatusCode: Int = -1 // default if onNext is not even invoked - error

        override def onNext(waitResponse: WaitResponse): Unit = {
          super.onNext(waitResponse)
          lastStatusCode = waitResponse.getStatusCode
        }

        override def onComplete(): Unit = {
          Try(super.onComplete()) match {
            case Failure(ex) =>
              handleWaitForContainerFailure(ex)
            case Success(_) =>
              submitCommand(DockerOperationSuccess(jobId, event = WaitForContainerCompleted(lastStatusCode)))
          }
        }

        override def onError(throwable: Throwable): Unit = {
          Try(super.onError(throwable)) match {
            case Failure(ex) =>
              handleWaitForContainerFailure(ex)
            case Success(_) =>
              throwable match {
                case e@(_: ConnectionClosedException | _: HttpClient4ConnectionClosedException | _: ConnectException | _: SocketException) =>
                  submitCommand(DockerOperationFailure(jobId, event = ConnectionFailed(e.getMessage)))
                case re: RuntimeException if re.getCause.isInstanceOf[ConnectException] =>
                  submitCommand(DockerOperationFailure(jobId, event = ConnectionFailed(re.getCause.getMessage)))
                case _ =>
                  handleWaitForContainerFailure(throwable)
              }
          }
        }

        override def actorRef: ActorRef = selfRef

        private def handleWaitForContainerFailure(ex: Throwable): Unit = {
          val errorMessage = s"Unable to fetch status code from container [$containerId] for jobId [$jobId]"
          log.error(ex, errorMessage)
          submitCommand(DockerOperationFailure(jobId, OperationFailed(s"$errorMessage: ${ex.getMessage}")))
        }
      })
    } catch {
      case ex: Exception =>
        val errorMessage = s"Unable to fetch status code from container [$containerId] for jobId [$jobId]"
        log.error(ex, errorMessage)
        selfRef ! DockerOperationFailure(jobId, OperationFailed(s"$errorMessage: ${ex.getMessage}"))
    }
  }

  def createContainerOutputContext(jobId: JobId, containerId: String): Unit = {
    Try {
      val resultStream = dockerClient.copyArchiveFromContainerCmd(containerId, "output").exec()
      Using.resource(new TarArchiveInputStream(resultStream)) { tar =>
        for {
          tarEntry <- Option(tar.getNextTarEntry)
          if tarEntry.getName == "output"
        } yield {
          StreamUtils.copyToString(tar, StandardCharsets.UTF_8)
        }
      }
    } match {
      case Failure(ex) =>
        val errorMessage = s"Unable to create a container output context for jobId [$jobId] with containerId [$containerId]"
        handleFailure(ex, jobId, errorMessage)
      case Success(os) => os match {
        case Some(res) => selfRef ! DockerOperationSuccess(jobId, CreateContainerOutputContextCompleted(res))
        case None => selfRef ! DockerOperationFailure(jobId, OperationFailed("No 'output' was read"))
      }
    }
  }

  def removeContainer(jobId: JobId, containerId: String, containerName: String): Unit = {
    Try {
      val filteredContainers = getContainersByName(containerName)
      val evaluatedId = if (filteredContainers.isEmpty) containerId else filteredContainers.head.getId
      if (Environment.isDevelopment && xlrConfig.development.keepContainer) {
        log.info(s"Skipping container [$containerName] removal for development mode")
      } else {
        dockerClient.removeContainerCmd(evaluatedId).exec()
      }
    } match {
      case Failure(ex) =>
        val errorMessage = s"Unable to remove the container [$containerId] for jobId [$jobId]"
        handleFailure(ex, jobId, errorMessage)
      case Success(_) =>
        selfRef ! DockerOperationSuccess(jobId, RemoveContainerCompleted)
    }
  }

  def terminateContainer(jobId: JobId, containerId: String, containerName: String, timeout: Int): Unit = {
    Try {
      val filteredContainers = getContainersByName(containerName)
      val evaluatedId = if (filteredContainers.isEmpty) containerId else filteredContainers.head.getId
      if (hasText(evaluatedId) && getContainerState(evaluatedId).getRunning) {
        dockerClient.stopContainerCmd(evaluatedId).withTimeout(timeout).exec()
      }
    } match {
      case Failure(ex) =>
        val errorMessage = s"Unable to stop the container [$containerId] for jobId [$jobId]"
        handleFailure(ex, jobId, errorMessage)
      case Success(_) =>
        selfRef ! DockerOperationSuccess(jobId, TerminateJobCompleted)
    }
  }

  private def handleFailure(ex: Throwable,
                            jobId: JobId,
                            errorMessage: String): Unit = {
    log.error(ex, errorMessage)
    ex match {
      case e@(_: ConnectionClosedException | _: HttpClient4ConnectionClosedException | _: ConnectException | _: SocketException) =>
        selfRef ! DockerOperationFailure(jobId, event = ConnectionFailed(e.getMessage))
      case re: RuntimeException if re.getCause.isInstanceOf[ConnectException] =>
        selfRef ! DockerOperationFailure(jobId, event = ConnectionFailed(re.getCause.getMessage))
      case _ =>
        selfRef ! DockerOperationFailure(jobId, OperationFailed(s"$errorMessage: ${ex.getMessage}"))
    }
  }

  def ping(jobId: JobId): Unit = {
    Try {
      dockerClient.pingCmd().exec()
    } match {
      case Failure(exception) => selfRef ! DockerPingResult(jobId, success = false)
      case Success(value) => selfRef ! DockerPingResult(jobId, success = true)
    }
  }

  private def getContainersByName(containerName: String): Seq[Container] = {
    dockerClient.listContainersCmd().withShowAll(true).withNameFilter(List(containerName).asJava).exec().asScala.toSeq
  }

  private def getContainerState(containerId: String): InspectContainerResponse#ContainerState = {
    dockerClient.inspectContainerCmd(containerId).exec().getState
  }
}

class DockerLogHelper(taskId: String, jobId: JobId, lastChunk: Long) {
  private var currentChunk: Long = lastChunk
  private var currentBufferSize: Int = 0
  private final val bufferSize: Int = 8192
  private val byteArrayOutputStream = new ByteArrayOutputStream(bufferSize * 2)
  private val outputStream = new BufferedOutputStream(byteArrayOutputStream, bufferSize)
  private var lastLogTimestamp = Instant.now().getEpochSecond
  private var lastFlushTimestamp = lastLogTimestamp

  /**
   * Appends payload to the current buffer and returns log entry if buffer is full or last flush duration is more than 2 seconds.
   */
  def appendEntry(payload: Array[Byte]): Option[LogEntry] = {
    lastLogTimestamp = Instant.now().getEpochSecond
    outputStream.write(payload)
    currentBufferSize += payload.length
    val durationSinceLastFlush = (lastLogTimestamp - lastFlushTimestamp).seconds
    if (currentBufferSize > bufferSize || durationSinceLastFlush >= 2.seconds) {
      Some(flushBuffer())
    } else {
      None
    }
  }

  def getCurrentChunk: Long = this.currentChunk

  def getLastLogTimestamp: Long = this.lastLogTimestamp

  /**
   * Forcefully flush the buffer
   */
  def flush(): LogEntry = {
    lastLogTimestamp = Instant.now().getEpochSecond
    flushBuffer()
  }

  def close(): Unit = {
    Try(outputStream.close())
    Try(byteArrayOutputStream.close())
  }

  private def flushBuffer(): LogEntry = {
    currentChunk += 1
    outputStream.flush()
    val currentBuffer = byteArrayOutputStream.toByteArray
    val lastLogEntryTimestamp = DateTimeFormatter.ISO_INSTANT.format(Instant.ofEpochSecond(lastLogTimestamp))
    byteArrayOutputStream.reset()
    currentBufferSize = 0
    lastFlushTimestamp = Instant.now().getEpochSecond
    LogEntry(taskId, jobId, currentChunk, lastLogEntryTimestamp, currentBuffer, URI_SCHEME_LOCAL_STORAGE)
  }

}

trait CommandSupport {
  private var commandSubmitted: Boolean = false

  def actorRef: ActorRef

  def submitCommand(dockerJobCommand: DockerJobCommand): Unit = {
    if (!commandSubmitted) {
      actorRef ! dockerJobCommand
      commandSubmitted = true
    }
  }
}
