package com.xebialabs.xlrelease.runner.docker.actors

import akka.actor.{Actor, ActorLogging}
import com.xebialabs.xlrelease.runner.docker.actors.DirectiveParser.{Directive, directiveStartRegex}
import com.xebialabs.xlrelease.storage.domain.LogEntry
import com.xebialabs.xlrelease.support.akka.SerializableMsg
import org.apache.commons.lang3.StringUtils

import java.nio.charset.StandardCharsets
import scala.util.matching.Regex.Groups

trait LogEntryParser {
  self: Actor with ActorLogging =>

  def parser: DirectiveParser

  def parse(logEntry: LogEntry): Seq[Directive] = {
    try {
      import scala.jdk.StreamConverters._
      val lines = new String(logEntry.payload, StandardCharsets.UTF_8).lines().toScala(List)
      for {
        line <- lines
        directive <- parser.parseLine(line)
      } yield directive
    } catch {
      case e: Throwable =>
        log.error(e, "Unable to parse directives in the log entry")
        Seq()
    }
  }
}


object DirectiveParser {
  final val directiveStartRegex = "##\\[start: ([^]]+)\\]".r

  case class Directive(name: String, payload: String) extends SerializableMsg

  def apply(): DirectiveParser = new DirectiveParser()
}

class DirectiveParser extends SerializableMsg {
  private var currentBuffer: String = ""
  private var currentDirectiveName: Option[String] = None
  private var currentDirectivePayload: Option[String] = None

  private def startDirective(directiveName: String): Unit = {
    currentDirectiveName = Option(directiveName)
  }

  def endDirective(): Option[Directive] = {
    val directive = (currentDirectiveName, currentDirectivePayload) match {
      case (Some(name), Some(payload)) => Some(Directive(name, payload.trim()))
      case _ => None
    }
    currentDirectiveName = None
    currentBuffer = "" // there can be max one directive start or end marker per line
    directive
  }

  def directiveEndMarker(): String = currentDirectiveName match {
    case Some(name) => s"##[end: $name]"
    case None => ""
  }

  def directiveStartMarker(): String = currentDirectiveName match {
    case Some(name) => s"##[start: $name]"
    case None => ""
  }

  def isInsideDirective(): Boolean = currentDirectiveName.isDefined

  def parseLine(logLine: String): Option[Directive] = {
    val line = removeLogTimestamp(logLine)
    directiveStartRegex.findFirstMatchIn(line) collect {
      case Groups(name) => startDirective(name)
    }
    if (isInsideDirective()) {
      currentBuffer += line + System.lineSeparator()
    }
    val directiveEndIndex = currentBuffer.indexOf(directiveEndMarker())
    if (directiveEndIndex > 0) {
      currentDirectivePayload = Some(StringUtils.substringBetween(currentBuffer, directiveStartMarker(), directiveEndMarker()))
      endDirective()
    } else {
      None
    }
  }

  // remove timestamp information from the log line if present
  private def removeLogTimestamp(logLine: String): String = {
    val dateFormat = "\\d{4}-[01]\\d-[0-3]\\dT[0-2]\\d:[0-5]\\d:[0-5]\\d.\\d+Z".r
    val lines = Option(logLine).getOrElse("").split("\\s+", 2)
    if (lines.length == 2 && dateFormat.matches(lines.head)) {
      lines.last
    } else {
      logLine
    }
  }
}
