#
# Copyright (c) 2019. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

VERSION="v.0.0.001"

import sys

from delphix.lib.DelphixSession import DelphixSession
from delphix.lib.GetReferences import find_obj_by_name
from delphixpy.web import database, environment, group, source

def find_all_databases_by_group_name(engine, group_name, exclude_js_container=False):
    """
    Easy way to quickly find databases by group name
    """
    # First search groups for the name specified and return its reference
    group_obj = find_obj_by_name(engine, group, group_name)
    if group_obj:
        databases = database.get_all(engine, group=group_obj.reference, no_js_container_data_source=exclude_js_container)
        return databases

def find_database_by_name_and_group_name(engine, group_name, database_name):
    '''Finds a database using its name and the dataset group it belongs to, calling find_all_databases_by_group_name
    as a helper function'''
    databases = find_all_databases_by_group_name(engine, group_name)
    print("Searching for database %s in group %s.\n" % (database_name, group_name))
    for each in databases:
        if each.name == database_name:
            print("Found database with reference %s.\n" % str(each.reference))
            return each
    print("Unable to find \"" + database_name + "\" in " + group_name + ". Exiting")
    sys.exit(1)

def find_source_by_name(engine, source_name):
    '''Finds the reference for a data source using the source's name'''
    print("Searching for database source with name %s.\n" % source_name)
    source_obj = find_obj_by_name(engine, source, source_name)
    if not source_obj:
        print("Did not find a source \"" + source_name + "\". Exiting")
        sys.exit(1)
    return source_obj

def find_source_by_database(engine, database_obj):
    '''Finds the reference for a '''
    # The source tells us if the database is enabled/disables, virtual, vdb/dSource, or is a staging database.
    source_obj = source.get_all(engine, database=database_obj.reference)
    # We'll just do a little sanity check here to ensure we only have a 1:1 result.
    if len(source_obj) == 0:
        print("Did not find a source for " + database_obj.name + ". Exiting")
        sys.exit(1)
    elif len(source_obj) > 1:
        print("More than one source returned for " + database_obj.name + ". Exiting")
        print(source_obj)
        sys.exit(1)
    return source_obj

def delete_database(engine, source_obj, container_obj, obj_type=None):
    """
    This function 
    FYI - Snapshot is also called sync
    """
    # Sanity check to make sure our source object has a reference
    if source_obj.reference != None :
        # If we specified the --object_type flag, ensure this source is a match. Skip, if not.
        if obj_type != None and ((obj_type.lower() == "vdb" and source_obj.virtual != True ) or (obj_type.lower() == "dsource" and source_obj.virtual != False )):
            print(container_obj.name + " is not a " + obj_type.lower() + ". Skipping sync")
        # Ensure this source is not a staging database. We can't act upon those.
        elif source_obj.staging == True:
            print(container_obj.name + " is a staging database. Skipping.")
        # Ensure the source is enabled. We can't snapshot disabled databases.
        else:
            print("Deleting " + container_obj.name )
            print("Type: " + source_obj.type )
            print(source_obj.type)
            # Delete it
            database.delete(engine, container_obj.reference)


def delphix_delete_database(engine):
    """
    This function is where the main workflow resides.
    """

    print("Getting environment for %s.\n" + environmentName)
    # Get the environment object by the hostname
    environment_obj = find_obj_by_name(engine, environment, environmentName)
    if environment_obj != None:
        # Get all the sources running on the server
        env_source_objs = source.get_all(engine, environment=environment_obj.reference)
        # If the server doesn't have any objects, exit.
        if env_source_objs == None:
            print(environmentName + "does not have any objects. Exiting")
            sys.exit(1)
    else:
        print("No environment found for " + environmentName + ". Exiting")
        sys.exit(1)

    # Get the database object from the name
    database_obj = find_database_by_name_and_group_name(engine, datasetGroupName, vdbName)

    # Get the source of the database.
    # The source tells us if the database is enabled/disables, virtual, vdb/dSource, or is a staging database.
    source_obj = find_source_by_database(engine, database_obj)[0]
    # If we applied the environment/server filter AND group filter, find the intersecting matches
    if environment_obj != None and datasetGroupName:
        match = False
        for env_source_obj in env_source_objs:
            if source_obj.reference in env_source_obj.reference:
                match = True
                break
        if not match:
            print(database_obj.name + " does not exist on " + environmentName + ". Exiting")
            sys.exit(1)
    delete_database(engine, source_obj, database_obj, "vdb")


# Connect to the Delphix server
print("Connecting to the Delphix server...\n")
engine = DelphixSession.create(server).server_session
delphix_delete_database(engine)
