#
# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

import json
import time
import org.slf4j.LoggerFactory as LoggerFactory
from continuousTesting.HttpRequest import HttpRequest as ContinuousTestingHttpRequest

logger = LoggerFactory.getLogger("ContinuousTesting")

def getManualReportSteps(serverParams, reportApiID):
    # setup the request url
    api_endpoint = "/reporter/api/tests?report_api_id=%s&includeSteps=true" % reportApiID
    logger.info("api_endpoint = %s" % api_endpoint)

    request = ContinuousTestingHttpRequest(serverParams)
    response = request.get(
        api_endpoint
    )

    if response.isSuccessful():
        data = json.loads(response.response)
        logger.info("BODY======================\n%s\n====================" % json.dumps(data,indent=4, sort_keys=True))
        return data
    else:
        logger.error('Error in method getManualReportSteps - status code is: %s' % response.getStatus)
        #logger.error('Error in method getManualReportSteps - dump is: %s' % response.errorDump())
        raise Exception("Error getting Manual Report Steps. Please check input parameters.")


def createManualReportTemplate(serverParams, deviceQuery, testName, testStepsList):
    # setup the request url
    api_endpoint = "/api/v1/devices/web-control"

    # Example of testStepsList:
    #
    # user_input = {'name': 'stepName', 'description': 'stepDescription'},
    testSteps = json.loads("[%s]" % testStepsList)
    payload = json.dumps(
        {
            'deviceQuery': deviceQuery,
            'testName': testName,
            'testSteps': testSteps
        },
         indent=4, 
         sort_keys=True
    )

    # define necessary headers
    headers = {
        'Content-Type': 'application/json',
        'Accept': '*/*',
        'Accept-Encoding': 'deflate'
    }

    request = ContinuousTestingHttpRequest(serverParams)
    response = request.post(
        api_endpoint,
        body=payload,
        headers=headers
    )

    logger.info("BODY======================\n%s\n====================" % payload)
    if response.isSuccessful():
        linkToOpenDevice = getJSONValueFromResponseContent('link', response.response)
        reportApiID = getJSONValueFromResponseContent('report_api_id', response.response)
        return response.response, linkToOpenDevice, reportApiID
    else:
        logger.error('Error in method createManualReportTemplate - status code is: %s' % response.getStatus)
        logger.error('Error in method createManualReportTemplate - dump is: %s' % response.errorDump())
        raise Exception("Error creating Manual Report Template. Please check input parameters.")


def triggerEspressoOrXCUITest(serverParams, deviceQueries, runningType, executionType, appUrl, testAppUrl):
    # setup the request url
    api_endpoint = "/api/v1/test-run/execute-test-run-async"

    # set the payload
    payload = json.dumps(
        {
            'executionType': executionType,
            'runningType': runningType,
            'deviceQueries': deviceQueries
        }
    )

    request = ContinuousTestingHttpRequest(serverParams)
    response = request.post(
        api_endpoint,
        body=payload,
        executionType=executionType,
        runningType=runningType,
        deviceQueries=deviceQueries,
        appUrl=appUrl,
        testAppUrl=testAppUrl
    )

    if response.isSuccessful():
        reporterUrlLink = getJSONValueFromResponseContent('Link to Reporter', response.response)
        testRunId = getJSONValueFromResponseContent('Test Run Id', response.response)
        return response.response, reporterUrlLink, testRunId
    else:
        logger.error('Error in method triggerEspressoOrXCUITest - status code is: %s' % response.getStatus)
        logger.error('Error in method triggerEspressoOrXCUITest - dump is: %s' % response.errorDump())
        raise Exception("Error initiating test case execution. Please check input parameters.")


def getTestRunStatusAndResultForUnitTests(serverParams, testRunId):
    # setup the request url
    api_endpoint = "/api/v1/test-run/%s/status" % testRunId

    request = ContinuousTestingHttpRequest(serverParams)
    response = request.get(api_endpoint)

    testRunStatus = getJSONValueFromResponseContent('Test Run State', response.response)

    while testRunStatus != "Finished" and "Running":
        # Add logic to allow for a timeout based run time of results
        # This should be a user input and allow them to decide the number of minutes to wait
        time.sleep(20)
        response = request.get(api_endpoint)
        testRunStatus = getJSONValueFromResponseContent('Test Run State', response.response)

    if response.isSuccessful():
        totalNumberOfTests = getJSONValueFromResponseContent('Total number of tests', response.response)
        passedCount = getJSONValueFromResponseContent('Number of passed tests', response.response)
        failedCount = getJSONValueFromResponseContent('Number of failed tests', response.response)
        skippedCount = getJSONValueFromResponseContent('Number of skipped tests', response.response)
        ignoredCount = getJSONValueFromResponseContent('Number of ignored tests', response.response)
        reporterLinkUrl = getJSONValueFromResponseContent('Link to Reporter', response.response)

        return testRunStatus, totalNumberOfTests, passedCount, failedCount, skippedCount, ignoredCount, reporterLinkUrl
    else:
        logger.error('Error in method getTestRunStatusAndResultForUnitTests - status code is: %s' % response.getStatus)
        logger.error('Error in method getTestRunStatusAndResultForUnitTests - dump is: %s' % response.errorDump())
        raise Exception("Error retrieving status and results. Please check input parameters.")


def createTestView(serverParams, testViewName):
    # setup the request url
    api_endpoint = "/reporter/api/testView"

    # build necessary payload
    payload = json.dumps(
        {
            'name': testViewName,
            'byKey': 'date',
            'groupByKey1': 'device.os',
            'groupByKey2': 'device.version',
            'showInDashboard': False,
            'viewBy': 'data'
        }
    )

    # define necessary headers
    headers = {
        'Content-Type': 'application/json',
        'Accept': '*/*',
        'Accept-Encoding': 'deflate'
    }

    # execute post request call
    request = ContinuousTestingHttpRequest(serverParams)
    response = request.post(
        api_endpoint,
        body=payload,
        headers=headers
    )

    if response.isSuccessful():
        testViewId = getTestViewId(response.response)
        return response.response, testViewId
    else:
        logger.error('Error in method createTestView - status code is: %s' % response.getStatus)
        logger.error('Error in method createTestView - dump is: %s' % response.errorDump())
        raise Exception("Error creating test view. Please check input parameters.")


def getTestViewResults(serverParams, testViewId, jenkinsBuildNumber):
    # setup the request url
    api_endpoint = "/reporter/api/testView/%s/summary" % testViewId
    if jenkinsBuildNumber is None:
        filter_api_endpoint = api_endpoint
    else:
        filter_api_endpoint = api_endpoint + "?filter={\"Jenkins_Build_Number\":\"%s\"}" % jenkinsBuildNumber

    request = ContinuousTestingHttpRequest(serverParams)
    response = request.get(filter_api_endpoint)

    if response.isSuccessful():
        totalTestCount = getAppiumJSONValueFromResponseContent('_count_', response.response)
        passedCount = getAppiumJSONValueFromResponseContent('passedCount', response.response)
        failedCount = getAppiumJSONValueFromResponseContent('failedCount', response.response)
        skippedCount = getAppiumJSONValueFromResponseContent('skippedCount', response.response)
        incompleteCount = getAppiumJSONValueFromResponseContent('incompleteCount', response.response)
        return totalTestCount, passedCount, failedCount, skippedCount, incompleteCount
    else:
        logger.error('Error in method getTestViewResults - status code is: %s' % response.getStatus)
        logger.error('Error in method getTestViewResults - dump is: %s' % response.errorDump())
        raise Exception("Error getting test view results. Please check input parameters.")


# This gets the given 'value' based on the response content which is structured as referenced here:
# https://docs.experitest.com/display/TE/Manage+Test+Run+with+the+API#ManageTestRunwiththeAPI-StatusoftheAPIRun
def getJSONValueFromResponseContent(value, responseContent):
    data = json.loads(responseContent)
    returnValue = ""

    for key in data:
        if key == "data":
            for subKey in data['data']:
                if subKey == "%s" % value:
                    returnValue = data['data']['%s' % value]
                    break
            break

    return returnValue


# The appium result set returns a JSON array object.  Need to get the first record in that array and parse it.
# There is only ever 1 record in the array
def getAppiumJSONValueFromResponseContent(value, responseContent):
    data = json.loads(responseContent)
    returnValue = ""

    for key in data:
        if key == "data":
            subData = data['data'][0]
            for subKey in subData:
                if subKey == "%s" % value:
                    returnValue = subData['%s' % value]
                    # break
            # break

    return returnValue


# This method returns the test view ID from the response
def getTestViewId(responseContent):
    testViewId = ""
    data = json.loads(responseContent)

    for key in data:
        if key == "id":
            testViewId = data['id']
            break

    return testViewId
