#
# Copyright 2024 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import json
from xlrelease.HttpRequest import HttpRequest
from confluence.util import error
from confluence.ConfluenceClient import ConfluenceClient

class ConfluenceDataCenterClient(ConfluenceClient):
    """
    Data Center-specific client that extends ConfluenceClient.
    Overrides methods to use Data Center v1 REST APIs instead of Cloud v2 APIs.
    """

    def addComment(self, pageId, comment):
        addCommentUrl = "/rest/api/content"
        payload = {
            "type": "comment",
            "container": {
                "id": pageId,
                "type": "page"
            },
            "body": {
                "storage": {
                    "representation": "storage",
                    "value": comment
                }
            }
        }

        response = self.httpRequest.post(addCommentUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 200 and response.getStatus() != 201:
            error("Failed to add comment", response)
        else:
            result = json.loads(response.response)
            commentId = result.get('id')
            print(result['_links']['base'] + result['_links']['webui'] + '\n')
            return commentId

    def addPage(self, spaceKey, parentPageId, pageTitle, pageText):
        addPageUrl = '/rest/api/content'
        pageText = pageText.replace('\\"', '"')
        payload = {
            "type": "page",
            "title": pageTitle,
            "space": {
                "key": spaceKey
            },
            "body": {
                "storage": {
                    "representation": "storage",
                    "value": pageText
                }
            }
        }
        if parentPageId:
           payload['ancestors'] = [{"id": parentPageId}]

        response = self.httpRequest.post(addPageUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while adding the page", response)
        result = json.loads(response.response)
        pageId = result['id']
        print(str(result['_links']['base'])+str(result['_links']['webui']))
        return pageId

    def getSpaceId(self, spaceKey):
        getSpaceIdUrl = '/rest/api/space/%s' % spaceKey
        response = self.httpRequest.get(getSpaceIdUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while fetching the Space ID", response)
        result = json.loads(response.response)
        spaceId = result['id']
        return spaceId

    def deletePage(self, pageId):
        deletePageUrl = '/rest/api/content/%s' % pageId
        response = self.httpRequest.delete(deletePageUrl, headers=self.headers)
        if response.getStatus() != 204:
            error("An unexpected error occurred while deleting the page", response)

    def getPage(self, pageId):
        getPageUrl = '/rest/api/content/%s?expand=version,space,body.storage' % pageId
        response = self.httpRequest.get(getPageUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while fetching the page", response)
        return json.loads(response.response)

    def getPageIdsByTitle(self, spaceKey, pageTitles):
       pageIdList = []
       for pageTitle in pageTitles:
           searchByPageTitleUrl = '/rest/api/content?spaceKey=%s&title=%s&type=page' % (spaceKey, pageTitle)
           response = self.httpRequest.get(searchByPageTitleUrl, headers=self.headers)
           if response.getStatus() != 200:
               error("An unexpected error occurred while fetching the Page ID", response)
           result = json.loads(response.response)
           if not result['results']:
               raise Exception("No results found for the specified page title: %s" % pageTitle)
           for page in result['results']:
               pageIdList.append(page['id'])
       return pageIdList

    def updatePage(self, pageId, pageTitle, pageText, updateMessage):
        currentPage = self.getPage(pageId)
        newVersionNumber = currentPage['version']['number'] + 1
        updatePageUrl = '/rest/api/content/%s' % pageId
        pageText = pageText.replace('\\"', '"')
        payload = {
            "id": pageId,
            "type": "page",
            "title": pageTitle,
            "space": {
                "key": currentPage['space']['key']
            },
            "body": {
                "storage": {
                    "representation": "storage",
                    "value": pageText
                }
            },
            "version": {
                "number": newVersionNumber
            }
        }
        if updateMessage:
           payload['version']['message'] = updateMessage

        response = self.httpRequest.put(updatePageUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while updating the page", response)

    def getCurrentUserId(self):
        getCurrentUserUrl = '/rest/api/user/current'
        response = self.httpRequest.get(getCurrentUserUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while fetching the current user", response)
        result = json.loads(response.response)
        # Return username for Data Center (for restrictions), not userKey
        return result.get('username')

    def addRestrictionToPage(self, pageId, userIds):
        addRestrictionstoPageUrl = '/rest/api/content/' + pageId + '/restriction'
        users = []
        for userId in userIds:
            users.append({
                "type": "known",
                "username": userId
            })
        payload = [
            {
                "operation": "read",
                "restrictions": {
                    "user": users
                }
            },
            {
                "operation": "update",
                "restrictions": {
                    "user": users
                }
            }
        ]
        response = self.httpRequest.put(addRestrictionstoPageUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while adding the restriction", response)

    def addContentWatcher(self, pageId, userId):
        addContentWatcherUrl = '/rest/api/user/watch/content/' + pageId + '?username=%s' % (userId)
        payload = {}
        response = self.httpRequest.post(addContentWatcherUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 200 and response.getStatus() != 204:
            error("An unexpected error occurred while adding the content watcher", response)
