#
# Copyright 2024 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import json
from xlrelease.HttpRequest import HttpRequest
from confluence.util import error

class ConfluenceClient(object):

    def __init__(self, httpConnection):
        self.httpRequest = HttpRequest(httpConnection)
        self.headers = {"Accept": "application/json", "Content-Type": "application/json"}

    @staticmethod
    def createClient(httpConnection, username=None, password=None):
        connection = dict(httpConnection)
        if username and password:
            connection['username'] = username
            connection['password'] = password
        elif password and connection.get('isDataCenter'):
            connection['password'] = password
        if connection.get('isDataCenter'):
            from confluence.ConfluenceDataCenterClient import ConfluenceDataCenterClient
            return ConfluenceDataCenterClient(connection)
        else:
            return ConfluenceClient(connection)

    def addComment(self, pageId, comment):
        addCommentUrl = '/api/v2/footer-comments'
        payload = {
            "pageId": pageId,
            "body": {
                "representation": "storage",
                "value": comment
            }
        }

        response = self.httpRequest.post(addCommentUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 201:
            error("An unexpected error occurred while adding the comment", response)
        result = json.loads(response.response)
        commentId = result['id']
        print(str(result['_links']['base']) + str(result['_links']['webui']) + '\n')

    def addPage(self, spaceKey, parentPageId, pageTitle, pageText):
        addPageUrl = '/api/v2/pages'
        spaceId = self.getSpaceId(spaceKey)
        pageText = pageText.replace('\\"', '"')
        payload = {
            "spaceId": spaceId,
            "status": "current",
            "title": pageTitle,
            "body": {
                "representation": "storage",
                "value": pageText
            }
        }
        if parentPageId:
           payload['parentId'] = parentPageId

        response = self.httpRequest.post(addPageUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while adding the page", response)
        result = json.loads(response.response)
        pageId = result['id']
        print(str(result['_links']['base'])+str(result['_links']['webui']))
        return pageId

    def getSpaceId(self, spaceKey):
        getSpaceIdUrl = '/api/v2/spaces?keys=%s' % spaceKey
        response = self.httpRequest.get(getSpaceIdUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while fetching the Space ID", response)
        result = json.loads(response.response)
        if not result['results']:
            raise Exception("No results found for the specified space key: %s" % spaceKey)
        spaceId = result['results'][0]['id']
        return spaceId

    def deletePage(self, pageId):
        deletePageUrl = '/api/v2/pages/%s' % pageId
        response = self.httpRequest.delete(deletePageUrl, headers=self.headers)
        if response.getStatus() != 204:
            error("An unexpected error occurred while deleting the page", response)

    def getPage(self, pageId):
        getPageUrl = '/api/v2/pages/%s' % pageId
        response = self.httpRequest.get(getPageUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while fetching the page", response)
        return json.loads(response.response)

    def getPageIdsByTitle(self, spaceKey, pageTitles):
       pageIdList = []
       for pageTitle in pageTitles:
           spaceId=self.getSpaceId(spaceKey)
           searchByPageTitleUrl = '/api/v2/pages?space-id=%s&title=%s' % (spaceId, pageTitle)
           response = self.httpRequest.get(searchByPageTitleUrl, headers=self.headers)
           if response.getStatus() != 200:
               error("An unexpected error occurred while fetching the Page ID", response)
           result = json.loads(response.response)
           if not result['results']:
               raise Exception("No results found for the specified page title: %s" % pageTitle)
           for page in result['results']:
               pageIdList.append(page['id'])
       return pageIdList

    def getPageHtmlByTitle(self, spaceKey, pageTitle):
        pageID = self.getPageIdsByTitle(spaceKey, pageTitle)[0]
        searchByPageTitleUrl = '/rest/api/content/' + pageID + '?expand=body.storage'

        response = self.httpRequest.get(searchByPageTitleUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while fetching the HTML content of the page", response)
        result = json.loads(response.response)
        htmlpage = result['body']['storage']['value']
        return htmlpage

    def updatePage(self, pageId, pageTitle, pageText, updateMessage):
        newVersionNumber = self.getPage(pageId)['version']['number'] + 1
        updatePageUrl = '/api/v2/pages/%s' % pageId
        pageText = pageText.replace('\\"', '"')
        payload = {
            "id": pageId,
            "status": "current",
            "title": pageTitle,
            "body": {
                "representation": "storage",
                "value": pageText
            },
            "version": {
                "number": newVersionNumber
            }
        }
        if updateMessage:
           payload['version']['message'] = updateMessage

        response = self.httpRequest.put(updatePageUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while updating the page", response)

    def addLabelsToPage(self, pageId, labels):
        addLabelstoPageUrl = 'rest/api/content/' + pageId + '/label'
        payload = []
        for label in labels:
            if label.find(':') == -1:
                prefix="global"
            else:
                prefix = label.split(':')[0]
                label = label.split(':')[1]              
            payload.append( 
            {
                "prefix": prefix,
                "name": label,
            }
            ) 
            response = self.httpRequest.post(addLabelstoPageUrl, json.dumps(payload), headers=self.headers)
            if response.getStatus() != 200:
                error("An unexpected error occurred while adding the label", response)

    def getCurrentUserId(self):
        getCurrentUserUrl = 'rest/api/user/current'
        response = self.httpRequest.get(getCurrentUserUrl, headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while fetching the current user", response)
        result = json.loads(response.response)
        return result['accountId']

    def addRestrictionToPage(self, pageId, userIds):
        addRestrictionstoPageUrl = 'rest/api/content/' + pageId + '/restriction'
        users = []
        for userId in userIds:
            users.append({ "accountId": userId })
        payload = {
            "results": [
                {
                    "operation": "read",
                    "restrictions": {
                        "user": users
                    }
                },
                {
                    "operation": "update",
                    "restrictions": {
                        "user": users
                    }
                }
            ]
        }
        response = self.httpRequest.post(addRestrictionstoPageUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 200:
            error("An unexpected error occurred while adding the restriction", response)

    def addContentWatcher(self, pageId, userId):
        addContentWatcherUrl = 'rest/api/user/watch/content/' + pageId + '?accountId=%s' % (userId)
        payload = {}
        response = self.httpRequest.post(addContentWatcherUrl, json.dumps(payload), headers=self.headers)
        if response.getStatus() != 204:
            error("An unexpected error occurred while adding the content watcher", response)