/*
 * Copyright (c) 2018. All rights reserved.
 *
 * This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
 */
'use strict';

(function () {

    var CheckmarxQueryTileViewController = function ($scope, CheckmarxQueryService, XlrTileHelper) {
        var vm = this;

        vm.tileConfigurationIsPopulated = tileConfigurationIsPopulated;

        function tileConfigurationIsPopulated() {
            return !_.isEmpty(vm.config.checkmarxServer);
        }

        $scope.filterByKey = function (item) {
            return (item.key !== 'Project Name' && item.key !== 'Scan Id' && item.key !== 'CxSAST Scan Date' && item.key !== 'Risk Level Score');
        };

        function load(config) {
            if (tileConfigurationIsPopulated()) {
                vm.loading = true;
                CheckmarxQueryService.executeQuery(vm.tile.id, config).then(
                    function (response) {
                        var cxSastData = response.data.data;
                        var lstData = [];
                        if (!cxSastData.error) {
                            $scope.xlrTile.title = vm.tile.title;
                            lstData.push({
                                key: 'Scan Id',
                                value: cxSastData['scanId'],
                                url: cxSastData['checkmarxUrl'] + '/CxWebClient/ViewerMain.aspx?scanId=' + cxSastData['scanId'] + '&ProjectID=' + cxSastData['projectId']
                            });
                            lstData.push({
                                key: 'Project Name',
                                value: cxSastData['projectName'],
                                url: cxSastData['checkmarxUrl'] + '/CxWebClient/portal#/projectState/' + cxSastData['projectId'] + '/Summary'
                            });
                            lstData.push({key: 'CxSAST Scan Date', value: cxSastData['statisticsCalculationDate']});
                            lstData.push({key: 'Risk Level Score', value: cxSastData['scanRiskSeverity']});
                            vm.result = {'lstData': lstData};

                            var categories = ['Info', 'Low', 'Medium', 'High'];
                            var keys = ['infoSeverity', 'lowSeverity', 'mediumSeverity', 'highSeverity'];
                            var counts = [];
                            keys.forEach(function (key) {
                                counts.push(cxSastData[key]);
                            });
                            vm.riskReportBarChartData = createEchartOptions(categories, counts);
                        } else {
                            var error = ''
                            if (cxSastData.error instanceof Array) {
                                for (var index = 0; index < cxSastData.error.length; index++) {
                                    error = error + cxSastData.error[index]['msg']
                                }
                            } else {
                                error = cxSastData.error
                            }
                            vm.result = {'lstData': lstData, 'error': error};
                        }

                    }
                ).finally(function () {
                    vm.loading = false;
                });
            }
        }

        function createEchartOptions(categories, counts) {
            var DEFAULT_MAX = 30;
            return {
                grid: {
                    top:    '10%',
                    bottom: '10%',
                    left:   '15%',
                    right:  '10%'
                },
                yAxis: {
                    type: 'category',
                    data: categories,
                    axisLine: {show: false},
                    axisTick: {show: false},
                    axisPointer: {show: false}
                },
                xAxis: {
                    show: false,
                    type: 'value',
                    max: function () {
                        var valArray = counts.slice(0);
                        valArray.push(DEFAULT_MAX);
                        return Math.max(...valArray);
                    },
                    axisLabel: {
                        show: false
                    }
                },
                series: [{
                    data: counts,
                    type: 'bar',
                    itemStyle: {
                        normal: {
                            color: function (params) {
                                var colorList = [
                                    '#08b153', '#fcd866', '#fd8d10', '#d94c3d'
                                ];
                                return colorList[params.dataIndex]
                            },
                            label: {
                                show: true,
                                position: 'right',
                                formatter: '{c}',
                                color: 'rgb(0,0,0)'
                            }
                        }
                    }
                }]
            };
        }


        function refresh() {
            load({params: {refresh: true}});
        }

        vm.$onInit = function () {
            if ($scope.xlrDashboard) {
                // summary page
                vm.release = $scope.xlrDashboard.release;
                vm.tile = $scope.xlrTile.tile;
                if (vm.tile.properties == null) {
                    vm.config = vm.tile.configurationProperties;
                } else {
                    // new style since 7.0
                    vm.config = vm.tile.properties;
                }
            }
            load();
        };
        vm.refresh = refresh;
    };

    CheckmarxQueryTileViewController.$inject = ['$scope', 'xlrelease.checkmarx.CheckmarxQueryService', 'XlrTileHelper'];

    var CheckmarxQueryService = function (Backend) {

        function executeQuery(tileId, config) {
            return Backend.get("tiles/" + tileId + "/data", config);
        }

        return {
            executeQuery: executeQuery
        };
    };

    CheckmarxQueryService.$inject = ['Backend'];

    angular.module('xlrelease.checkmarx.tile', []);
    angular.module('xlrelease.checkmarx.tile').service('xlrelease.checkmarx.CheckmarxQueryService', CheckmarxQueryService);
    angular.module('xlrelease.checkmarx.tile').controller('checkmarx.CheckmarxQueryTileViewController', CheckmarxQueryTileViewController);
    angular.module('xlrelease').directive("addScrollBar", function () {
        return {
            link: function (scope, element) {
                element.parent().css('overflowY', 'auto')
            }
        }
    });

})();

