#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries and licensors.
#

import json
import urllib

from checkmarx.client.CheckmarxClientUtil import CheckmarxClientUtil


class CheckmarxService(object):
    HIGH_RISK = 3
    MEDIUM_RISK = 2
    LOW_RISK = 1
    UNKNOWN_RISK = 0
    MAX_ITEMS = 1000000

    @staticmethod
    def get_team_id_by_name(checkmarx_server, team_name):
        all_teams = CheckmarxService.get_teams(checkmarx_server)
        for team in all_teams:
            trunc_team_name = team["fullName"].replace("/", "", 1)
            if trunc_team_name.lower() == team_name.lower():
                return team["id"]
        raise Exception("Could not resolve team ID for team name: %s" % team_name)

    @staticmethod
    def resolve_project(checkmarx_server, project_name, team_id):
        project_list = CheckmarxService.get_project(checkmarx_server, project_name, team_id)
        if project_list:
            return project_list[0]["id"]
        else:
            project = CheckmarxService.create_default_project(checkmarx_server, project_name, team_id)
            return project["id"]

    @staticmethod
    def get_preset_id_by_name(checkmarx_server, preset_name):
        all_preset = CheckmarxService.get_presets(checkmarx_server)
        for preset in all_preset:
            if preset["name"].lower() == preset_name.lower():
                return preset["id"]
        raise Exception("Could not resolve preset ID for preset name: %s" % preset_name)

    @staticmethod
    def get_config_id_by_name(checkmarx_server, config_name):
        all_config = CheckmarxService.get_engine_configurations(checkmarx_server)
        for config in all_config:
            if config["name"].lower() == config_name.lower():
                return config["id"]
        raise Exception("Could not resolve configuration ID for configuration: %s" % config_name)

    @staticmethod
    def get_teams(checkmarx_server):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server, context="/CxRestAPI/auth/teams",
                                                     contentType="application/json")
        error_message = "Failed to get Teams. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_presets(checkmarx_server, is_checkmarx_one=False):
        if is_checkmarx_one:
            checkmarx_response = CheckmarxClientUtil.get(checkmarx_server, context="/api/presets",
                                                             contentType="application/json")
        else:
            checkmarx_response = CheckmarxClientUtil.get(checkmarx_server, context="/CxRestAPI/sast/presets",
                                                     contentType="application/json")
        error_message = "Failed to get Presets. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_project(checkmarx_server, project_name, team_id):
        url = "/CxRestAPI/projects?projectName={}&teamId={}".format(project_name, team_id)
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server, context=url, contentType="application/json")
        if checkmarx_response.getStatus() == 404:
            return None
        elif not checkmarx_response.isSuccessful():
            raise Exception("Could not get project data for : %s" % project_name)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_sca_project(checkmarx_server, project_name):
        url = "/risk-management/projects?name={}".format(project_name)
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server, context=url, contentType="application/json")
        if checkmarx_response.getStatus() == 404:
            return None
        elif not checkmarx_response.isSuccessful():
            raise Exception("Could not get project data for : %s" % project_name)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def create_default_project(checkmarx_server, project_name, team_id, is_checkmarx_one=False):
        context_checkmarx = "/CxRestAPI/projects"
        body = {
            "name": project_name,
            "owningTeam": team_id,
            "isPublic": True
        }
        if is_checkmarx_one:
            context_checkmarx = "/api/projects"
            body.pop('owningTeam', None)
            body.pop('isPublic', None)

        checkmarx_response = CheckmarxClientUtil.post(checkmarx_server, context=context_checkmarx, body=body,
                                                      contentType="application/json")
        error_message = "Failed to create default project. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_engine_configurations(checkmarx_server):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server, context="/CxRestAPI/sast/engineConfigurations",
                                                     contentType="application/json")
        error_message = "Failed to get Configurations. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def set_project_settings(checkmarx_server, project_id, preset_id, engine_configuration_id):
        body = {
            "projectId": project_id,
            "presetId": preset_id,
            "engineConfigurationId": engine_configuration_id
        }
        checkmarx_response = CheckmarxClientUtil.post(checkmarx_server, context="/CxRestAPI/sast/scanSettings",
                                                      body=body, contentType="application/json")
        error_message = "Failed to set project settings. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def set_remote_settings(checkmarx_server, project_id, body, source_control):
        checkmarx_response = CheckmarxClientUtil.post(checkmarx_server,
                                                      context="/CxRestAPI/projects/{}/sourceCode/remoteSettings/{}".format(
                                                          project_id, source_control),
                                                      body=body, contentType="application/json")
        error_message = "Failed to set Remote Settings. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return checkmarx_response.getStatus()

    @staticmethod
    def trigger_scan(checkmarx_server, project_id):
        body = {
            "projectId": project_id,
            "isIncremental": False,
            "isPublic": True,
            "forceScan": False
        }
        checkmarx_response = CheckmarxClientUtil.post(checkmarx_server,
                                                      context="/CxRestAPI/sast/scans",
                                                      body=body, contentType="application/json")
        error_message = "Failed to trigger scan. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())
    @staticmethod
    def checkmarxone_trigger_scan(checkmarx_server, body):
        checkmarx_response = CheckmarxClientUtil.post(checkmarx_server,
                                                      context="/api/scans/",
                                                      body=body, contentType="application/json")
        error_message = "Failed to trigger scan. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())
    @staticmethod
    def get_scan_status(checkmarx_server, scan_id, is_checkmarx_one=False):
        if is_checkmarx_one:
            checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                         context="/api/scans/{}".format(scan_id),
                                                         contentType="application/json")
        else:
            checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                         context="/CxRestAPI/sast/scans/{}".format(scan_id),
                                                         contentType="application/json")
        error_message = "Failed to get scan status. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_latest_sast_result_stats(checkmarx_server, project_id, is_checkmarx_one=False):
        scan_id = CheckmarxService.get_latest_sast_scan_id(checkmarx_server, project_id, is_checkmarx_one)
        return CheckmarxService.get_sast_results_stats(checkmarx_server, scan_id, is_checkmarx_one)

    @staticmethod
    def get_latest_sast_scan_id(checkmarx_server, project_id, is_checkmarx_one=False):
        if is_checkmarx_one:
            checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                         'api/scans/?offset=0&limit=20&sort=+created_at&field=scan-ids&project-names={}'.format(
                                                             project_id))
        else:
            checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                         '/CxRestAPI/sast/scans?projectId={}&last=1&scanStatus=Finished'.format(
                                                             project_id))
        error_message = "Failed to get CxSAST Status. Server return [%s], with content [%s]" % (
            checkmarx_response.status, checkmarx_response.response)
        CheckmarxService.validate_response(error_message, checkmarx_response)
        if is_checkmarx_one:
            return json.loads(checkmarx_response.getResponse())["scans"][0]["id"]
        else:
            return json.loads(checkmarx_response.getResponse())[0]["id"]

    @staticmethod
    def check_project_name_or_create(checkmarx_server,project_name=None):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                     'api/projects/?name={}'.format(project_name))
        error_message = "Failed to get Project List. Server return [%s], with content [%s]" % (
            checkmarx_response.status, checkmarx_response.response)
        CheckmarxService.validate_response(error_message, checkmarx_response)
        # Create if project name does not exist
        if json.loads(checkmarx_response.getResponse())["filteredTotalCount"] == 0:
            project = CheckmarxService.create_default_project(checkmarx_server, project_name, None,is_checkmarx_one=True)
            return project["id"]
        else:
            for project in json.loads(checkmarx_response.getResponse())["projects"]:
                if project["name"] == project_name:
                    return project["id"]
            raise Exception("Failed to get Project ID for project name: %s" % project_name)

    @staticmethod
    def get_projects_list(checkmarx_server):
        project_context = 'api/projects'
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                     project_context)
        error_message = "Failed to get Project List. Server return [%s], with content [%s]" % (
            checkmarx_response.status, checkmarx_response.response)
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_sast_results_stats(checkmarx_server, scan_id, is_checkmarx_one=False):
        error_message = ""
        if is_checkmarx_one:
            checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                         '/api/scan-summary/?scan-ids={}&include-queries=false&include-files=false'.format(scan_id))
            error_message = "Failed to get CheckmarxOne SAST Result Statistics."

        else:
            checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                         '/CxRestAPI/sast/scans/{}/resultsStatistics'.format(scan_id))
            error_message = "Failed to get CxSAST Result Statistics."
        error_message = error_message +" Server return [%s], with content [%s]" % (
            checkmarx_response.status, checkmarx_response.response)
        CheckmarxService.validate_response(error_message, checkmarx_response)
        response = json.loads(checkmarx_response.getResponse())
        response["scanId"] = scan_id
        return response

    @staticmethod
    def get_latest_osa_scan(checkmarx_server, project_id):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                     '/CxRestAPI/osa/scans?projectId={}'.format(project_id))
        error_message = "Failed to get CxOSA Status. Server return [%s], with content [%s]" % (
            checkmarx_response.status, checkmarx_response.response)
        CheckmarxService.validate_response(error_message, checkmarx_response)
        json_data = json.loads(checkmarx_response.getResponse())
        for json_obj in json_data:
            if json_obj["state"]["name"] == "Succeeded":
                return json_obj
        raise Exception("Failed to get CxOSA Scan with Succeeded status")

    @staticmethod
    def get_osa_scan_status(checkmarx_server, scan_id):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                     context="/CxRestAPI/osa/scans/{}".format(scan_id),
                                                     contentType="application/json")
        error_message = "Failed to get scan status. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_sca_risk_reports(checkmarx_server, project_id):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                     context="/risk-management/risk-reports?projectId={}&size=1".format(project_id),
                                                     contentType="application/json")
        error_message = "Failed to get risk reports. Server return [%s], with content [%s]" % (
            checkmarx_response.getStatus(), checkmarx_response.getResponse())
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_osa_results_stats(checkmarx_server, scan_id):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                     '/CxRestAPI/osa/reports?scanId={}&reportFormat=json'.format(
                                                         scan_id))
        error_message = "Failed to get CxOSA Result Statistics. Server return [%s], with content [%s]" % (
            checkmarx_response.status, checkmarx_response.response)
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_osa_libraries(checkmarx_server, scan_id):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                     '/CxRestAPI/osa/libraries?scanId={}&itemsPerPage={}'.format(scan_id, CheckmarxService.MAX_ITEMS))
        error_message = "Failed to get CxOSA Scan Libraries. Server return [%s], with content [%s]" % (
            checkmarx_response.status, checkmarx_response.response)
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_osa_licenses(checkmarx_server, scan_id):
        checkmarx_response = CheckmarxClientUtil.get(checkmarx_server,
                                                     '/CxRestAPI/osa/licenses?scanId={}'.format(scan_id))
        error_message = "Failed to get CxOSA Scan Licenses. Server return [%s], with content [%s]" % (
            checkmarx_response.status, checkmarx_response.response)
        CheckmarxService.validate_response(error_message, checkmarx_response)
        return json.loads(checkmarx_response.getResponse())

    @staticmethod
    def get_osa_libraries_data(checkmarx_server, scan_id):
        libraries = CheckmarxService.get_osa_libraries(checkmarx_server=checkmarx_server, scan_id=scan_id)
        licenses_list = CheckmarxService.get_osa_licenses(checkmarx_server=checkmarx_server, scan_id=scan_id)
        licenses = dict((license["id"], license) for license in licenses_list)
        outdated_lib_count, license_risk_high_count, license_risk_medium_count, license_risk_low_count, license_unknown_count = 0, 0, 0, 0, 0
        for library in libraries:
            if library["outdated"]:
                outdated_lib_count += 1
            risk_levels = [licenses[license]["riskLevel"] for license in library["licenses"]]
            if "High" in risk_levels:
                license_risk_high_count += 1
            elif "Medium" in risk_levels:
                license_risk_medium_count += 1
            elif "Low" in risk_levels:
                license_risk_low_count += 1
            else:
                license_unknown_count += 1
        return {"outdated": outdated_lib_count,
                "highRisk": license_risk_high_count,
                "mediumRisk": license_risk_medium_count,
                "lowRisk": license_risk_low_count,
                "unknownRisk": license_unknown_count}

    @staticmethod
    def validate_response(error_message, checkmarx_response):
        if not checkmarx_response.isSuccessful():
            raise Exception(error_message)

    @staticmethod
    def format_url(url, user=None, user_pass=None, user_token=None):
        encoded_url = url
        if user and user_pass:
            encoded_url = CheckmarxService.encode_url_with_cred(url=url,
                                                                cred="{}:{}".format(user, user_pass))
        elif user_token:
            encoded_url = CheckmarxService.encode_url_with_cred(url=url, cred=user_token)
        return encoded_url

    @staticmethod
    def encode_url_with_cred(url, cred):
        encoded_url = ""
        if url.startswith("https://"):
            encoded_url = url.replace("https://", "https://{}@".format(cred), 1)
        elif url.startswith("http://"):
            encoded_url = url.replace("http://", "http://{}@".format(cred), 1)
        return encoded_url

    @staticmethod
    def create_remote_git_body(url, branch, private_key=None):
        body = {
            "branch": branch,
            "url": url
        }
        if private_key:
            body["privateKey"] = private_key
        return body

    @staticmethod
    def create_checkmarxone_scan_git_body(url, branch, preset, sast, sca, user, user_pass, token, skip_sub_module, project_id, project_tags, scan_tags):
        credentials = {}
        if user_pass:
            credentials = { "username": user, "type": "password", "value": user_pass }
        elif token:
            credentials = { "username": user, "type": "apiKey", "value": token }

        body = {
            "type": "git",
            "handler": {
                "branch": branch,
                "repoUrl": url,
                "credentials": credentials,
                "skipSubModules": skip_sub_module
            },
            "project": {
                "id": project_id
            },
            "config": []
        }
        if sast:
            sast_object = {
                "type": "sast",
                "value": {
                    "incremental": "false"
                }
            }
            if preset:
                sast_object["value"]["preset"] = preset
            body["config"].append(sast_object)
        if sca:
            sca_object = {
                "type": "sca",
                "value": {}
            }
            body["config"].append(sca_object)
        if project_tags:
            body["project"]["tags"] = project_tags
        if scan_tags:
            body["tags"] = scan_tags
        return body

    @staticmethod
    def create_remote_svn_body(url, port, paths, user=None, user_pass=None, private_key=None):
        body = {
            "uri": {
                "absoluteUrl": url,
                "port": port
            },
            "paths": paths
        }
        if user and user_pass:
            body["credentials"] = {
                "userName": user,
                "password": user_pass
            }
        elif private_key:
            body["privateKey"] = private_key
        return body
